<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('users', function (Blueprint $table) {
            // Add missing fields
            if (! Schema::hasColumn('users', 'directory_visible')) {
                $table->boolean('directory_visible')->default(true)->after('status');
            }

            if (! Schema::hasColumn('users', 'last_login_at')) {
                $table->timestamp('last_login_at')->nullable()->after('directory_visible');
            }

            // Update status enum to include 'suspended'
            if (Schema::hasColumn('users', 'status')) {
                // SQLite doesn't support ALTER COLUMN for enum, so we'll need to recreate
                // For now, we'll keep the existing enum and add a note
            }

            // Add soft deletes if not exists
            if (! Schema::hasColumn('users', 'deleted_at')) {
                $table->softDeletes();
            }

            // Add indexes for performance
            if (! $this->hasIndex('users', 'users_role_id_status_index')) {
                try {
                    $table->index(['role_id', 'status'], 'users_role_id_status_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }
            }

            if (! $this->hasIndex('users', 'users_society_id_status_index')) {
                try {
                    $table->index(['society_id', 'status'], 'users_society_id_status_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }
            }

            if (! $this->hasIndex('users', 'users_fcm_token_index')) {
                try {
                    $table->index('fcm_token', 'users_fcm_token_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }
            }

            if (! $this->hasIndex('users', 'users_role_id_society_id_index')) {
                try {
                    $table->index(['role_id', 'society_id'], 'users_role_id_society_id_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('users', function (Blueprint $table) {
            if (Schema::hasColumn('users', 'directory_visible')) {
                $table->dropColumn('directory_visible');
            }

            if (Schema::hasColumn('users', 'last_login_at')) {
                $table->dropColumn('last_login_at');
            }

            if (Schema::hasColumn('users', 'deleted_at')) {
                $table->dropSoftDeletes();
            }

            try {
                $table->dropIndex('users_role_id_status_index');
            } catch (\Exception $e) {
                // Index may not exist
            }

            try {
                $table->dropIndex('users_society_id_status_index');
            } catch (\Exception $e) {
                // Index may not exist
            }

            try {
                $table->dropIndex('users_fcm_token_index');
            } catch (\Exception $e) {
                // Index may not exist
            }

            try {
                $table->dropIndex('users_role_id_society_id_index');
            } catch (\Exception $e) {
                // Index may not exist
            }
        });
    }

    /**
     * Check if index exists (simplified for SQLite compatibility)
     */
    private function hasIndex(string $table, string $index): bool
    {
        try {
            $connection = Schema::getConnection();
            $doctrineSchemaManager = $connection->getDoctrineSchemaManager();
            $doctrineTable = $doctrineSchemaManager->introspectTable($table);

            return $doctrineTable->hasIndex($index);
        } catch (\Exception $e) {
            return false;
        }
    }
};
