<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('societies', function (Blueprint $table) {
            // Add new required fields
            if (! Schema::hasColumn('societies', 'registration_no')) {
                $table->string('registration_no')->unique()->after('name');
            }

            // Update address to be required (if it was nullable)
            if (Schema::hasColumn('societies', 'address')) {
                $table->text('address')->nullable(false)->change();
            }

            // Update city, state, pincode to be required
            if (Schema::hasColumn('societies', 'city')) {
                $table->string('city')->nullable(false)->change();
            }
            if (Schema::hasColumn('societies', 'state')) {
                $table->string('state', 50)->nullable(false)->change();
            }
            if (Schema::hasColumn('societies', 'pincode')) {
                $table->string('pincode', 10)->nullable(false)->change();
            }

            // Add platform_user_id if not exists
            if (! Schema::hasColumn('societies', 'platform_user_id')) {
                $table->foreignId('platform_user_id')->nullable()->after('pincode')->constrained('users')->nullOnDelete();
            }

            // Add financial fields
            if (! Schema::hasColumn('societies', 'platform_fee_amount')) {
                $table->decimal('platform_fee_amount', 10, 2)->default(2500)->after('platform_user_id');
            }
            if (! Schema::hasColumn('societies', 'platform_fee_frequency')) {
                $table->enum('platform_fee_frequency', ['monthly', 'quarterly', 'annual'])->default('monthly')->after('platform_fee_amount');
            }

            // Update status enum
            if (Schema::hasColumn('societies', 'status')) {
                $table->enum('status', ['pending', 'active', 'suspended', 'inactive'])->default('pending')->change();
            }

            // Add branding & docs fields
            if (! Schema::hasColumn('societies', 'logo')) {
                $table->string('logo')->nullable()->after('status');
            }
            if (! Schema::hasColumn('societies', 'gst_no')) {
                $table->string('gst_no')->nullable()->after('logo');
            }
            if (! Schema::hasColumn('societies', 'pan_no')) {
                $table->string('pan_no')->nullable()->after('gst_no');
            }
            if (! Schema::hasColumn('societies', 'rules')) {
                $table->text('rules')->nullable()->after('pan_no');
            }

            // Add soft deletes if not exists
            if (! Schema::hasColumn('societies', 'deleted_at')) {
                $table->softDeletes();
            }

            // Add indexes (skip for SQLite to avoid duplicate index errors in tests)
            if (DB::getDriverName() !== 'sqlite') {
                try {
                    $table->index(['status', 'created_at'], 'societies_status_created_at_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }

                try {
                    $table->index('platform_user_id', 'societies_platform_user_id_index');
                } catch (\Exception $e) {
                    // Index may already exist
                }
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('societies', function (Blueprint $table) {
            if (Schema::hasColumn('societies', 'registration_no')) {
                $table->dropUnique(['registration_no']);
                $table->dropColumn('registration_no');
            }

            $columnsToDrop = [
                'platform_user_id',
                'platform_fee_amount',
                'platform_fee_frequency',
                'logo',
                'gst_no',
                'pan_no',
                'rules',
            ];

            foreach ($columnsToDrop as $column) {
                if (Schema::hasColumn('societies', $column)) {
                    $table->dropColumn($column);
                }
            }

            if (Schema::hasColumn('societies', 'deleted_at')) {
                $table->dropSoftDeletes();
            }

            try {
                $table->dropIndex('societies_status_created_at_index');
            } catch (\Exception $e) {
                // Index may not exist
            }

            try {
                $table->dropIndex('societies_platform_user_id_index');
            } catch (\Exception $e) {
                // Index may not exist
            }
        });
    }
};
