<?php

namespace App\Services;

use App\Models\BankAccount;
use App\Models\Transaction;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class TransactionService
{
    /**
     * Record an income (Credit) transaction.
     */
    public function recordIncome(
        float $amount,
        int $bankAccountId,
        string $category,
        ?Model $reference = null,
        string $paymentMethod = 'online',
        ?string $description = null,
        ?int $createdBy = null
    ): Transaction {
        return DB::transaction(function () use ($amount, $bankAccountId, $category, $reference, $paymentMethod, $description, $createdBy) {
            $bankAccount = BankAccount::lockForUpdate()->find($bankAccountId);

            if (! $bankAccount) {
                throw new \Exception("Bank account not found.");
            }

            $bankAccount->balance += $amount;
            $bankAccount->save();

            $transaction = Transaction::create([
                'society_id' => $bankAccount->society_id,
                'bank_account_id' => $bankAccountId,
                'type' => 'credit',
                'amount' => $amount,
                'category' => $category,
                'reference_type' => $reference ? get_class($reference) : null,
                'reference_id' => $reference ? $reference->id : null,
                'description' => $description,
                'transaction_date' => now(),
                'payment_method' => $paymentMethod,
                'created_by' => $createdBy ?? auth()->id(),
                'balance_after' => $bankAccount->balance,
            ]);

            return $transaction;
        });
    }

    /**
     * Record an expense (Debit) transaction.
     */
    public function recordExpense(
        float $amount,
        int $bankAccountId,
        string $category,
        ?Model $reference = null,
        string $paymentMethod = 'online',
        ?string $description = null,
        ?int $createdBy = null
    ): Transaction {
        return DB::transaction(function () use ($amount, $bankAccountId, $category, $reference, $paymentMethod, $description, $createdBy) {
            $bankAccount = BankAccount::lockForUpdate()->find($bankAccountId);

            if (! $bankAccount) {
                throw new \Exception("Bank account not found.");
            }

            $bankAccount->balance -= $amount;
            $bankAccount->save();

            $transaction = Transaction::create([
                'society_id' => $bankAccount->society_id,
                'bank_account_id' => $bankAccountId,
                'type' => 'debit',
                'amount' => $amount,
                'category' => $category,
                'reference_type' => $reference ? get_class($reference) : null,
                'reference_id' => $reference ? $reference->id : null,
                'description' => $description,
                'transaction_date' => now(),
                'payment_method' => $paymentMethod,
                'created_by' => $createdBy ?? auth()->id(),
                'balance_after' => $bankAccount->balance,
            ]);

            return $transaction;
        });
    }

    /**
     * Handle Cash Handover from Collector to Bank.
     */
    public function handleCashHandover(int $collectorId, int $bankAccountId, float $amount, ?string $notes = null): Transaction
    {
        // 1. Credit the Bank Account
        $transaction = $this->recordIncome(
            $amount,
            $bankAccountId,
            'cash_handover',
            null, // No direct reference model, could reference the User (Collector) but created_by handles it
            'cash',
            $notes ?? "Cash handover from Collector ID: $collectorId",
            $collectorId // The collector "created" this revenue flow effectively
        );

        // 2. Mark pending cash payments as 'handed_over' or similar?
        // Actually, we don't have a 'handed_over' status on payments.
        // We calculate "Cash in Hand" dynamically: Sum(Cash Payments) - Sum(Handovers).
        // This Service just logs the handover event. The Controller will use this to show remaining balance.

        return $transaction;
    }
}
