<?php

namespace App\Services;

use App\Mail\Resident\BillDueReminder;
use App\Mail\Resident\BillGenerated;
use App\Mail\Resident\BillOverdue;
use App\Mail\Resident\PaymentConfirmation;
use App\Models\Bill;
use App\Models\Payment;
use App\Models\Society;
use App\Models\SocietySetting;
use App\Models\User;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class ResidentEmailService
{
    /**
     * Configure mailer with society's SMTP settings.
     */
    protected static function configureSocietyMailer(SocietySetting $settings): bool
    {
        if (!$settings->email_enabled) {
            return false;
        }

        // Check if SMTP is configured
        if (empty($settings->smtp_host) || empty($settings->from_email)) {
            return false;
        }

        // Dynamically configure the mailer for this society
        Config::set('mail.mailers.society', [
            'transport' => 'smtp',
            'host' => $settings->smtp_host,
            'port' => $settings->smtp_port ?? 587,
            'encryption' => $settings->smtp_encryption ?? 'tls',
            'username' => $settings->smtp_username,
            'password' => $settings->smtp_password,
            'timeout' => null,
        ]);

        Config::set('mail.from.address', $settings->from_email);
        Config::set('mail.from.name', $settings->from_name ?? $settings->society->name);

        return true;
    }

    /**
     * Send bill generated email to resident.
     */
    public static function sendBillGeneratedEmail(Bill $bill): void
    {
        $resident = $bill->user;
        if (!$resident || empty($resident->email)) {
            return;
        }

        $society = $bill->society;
        $settings = $society->settings;

        // Skip if email not configured for society
        if (!$settings || !self::configureSocietyMailer($settings)) {
            return;
        }

        try {
            Mail::mailer('society')
                ->to($resident->email)
                ->queue(new BillGenerated($bill, $resident, $society));
        } catch (\Exception $e) {
            Log::error('Failed to send bill generated email', ['error' => $e->getMessage(), 'bill_id' => $bill->id]);
        }
    }

    /**
     * Send bill due reminder email to resident.
     */
    public static function sendBillDueReminderEmail(Bill $bill, int $daysUntilDue): void
    {
        $resident = $bill->user;
        if (!$resident || empty($resident->email)) {
            return;
        }

        $society = $bill->society;
        $settings = $society->settings;

        if (!$settings || !self::configureSocietyMailer($settings)) {
            return;
        }

        try {
            Mail::mailer('society')
                ->to($resident->email)
                ->queue(new BillDueReminder($bill, $resident, $society, $daysUntilDue));
        } catch (\Exception $e) {
            Log::error('Failed to send bill due reminder', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Send payment confirmation email to resident.
     */
    public static function sendPaymentConfirmationEmail(Payment $payment): void
    {
        $bill = $payment->bill;
        if (!$bill) {
            return;
        }

        $resident = $payment->user ?? $bill->user;
        if (!$resident || empty($resident->email)) {
            return;
        }

        $society = $payment->society ?? $bill->society;
        $settings = $society->settings;

        if (!$settings || !self::configureSocietyMailer($settings)) {
            return;
        }

        try {
            Mail::mailer('society')
                ->to($resident->email)
                ->queue(new PaymentConfirmation($payment, $bill, $resident, $society));
        } catch (\Exception $e) {
            Log::error('Failed to send payment confirmation', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Send bill overdue email to resident.
     */
    public static function sendBillOverdueEmail(Bill $bill, int $daysOverdue): void
    {
        $resident = $bill->user;
        if (!$resident || empty($resident->email)) {
            return;
        }

        $society = $bill->society;
        $settings = $society->settings;

        if (!$settings || !self::configureSocietyMailer($settings)) {
            return;
        }

        try {
            Mail::mailer('society')
                ->to($resident->email)
                ->queue(new BillOverdue($bill, $resident, $society, $daysOverdue));
        } catch (\Exception $e) {
            Log::error('Failed to send bill overdue email', ['error' => $e->getMessage()]);
        }
    }
}
