<?php

namespace App\Services\PaymentGateway;

use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;

class StripeGateway implements PaymentGatewayInterface
{
    protected string $publicKey;
    protected string $secretKey;
    protected bool $enabled;

    public function __construct(string $publicKey, string $secretKey, bool $enabled = true)
    {
        $this->publicKey = $publicKey;
        $this->secretKey = $secretKey;
        $this->enabled = $enabled;

        if ($this->isEnabled()) {
            Stripe::setApiKey($this->secretKey);
        }
    }

    public function getIdentifier(): string
    {
        return 'stripe';
    }

    public function getName(): string
    {
        return 'Stripe';
    }

    public function isEnabled(): bool
    {
        return $this->enabled && !empty($this->publicKey) && !empty($this->secretKey);
    }

    public function getPublicKey(): ?string
    {
        return $this->publicKey;
    }

    public function createOrder(float $amount, string $currency, array $metadata = []): array
    {
        try {
            // Create PaymentIntent for client-side confirmation
            $paymentIntent = PaymentIntent::create([
                'amount' => (int) ($amount * 100), // Convert to cents/paise
                'currency' => strtolower($currency),
                'metadata' => $metadata['notes'] ?? [],
                'automatic_payment_methods' => [
                    'enabled' => true,
                ],
            ]);

            return [
                'gateway' => $this->getIdentifier(),
                'order_id' => $paymentIntent->id,
                'client_secret' => $paymentIntent->client_secret,
                'amount' => $paymentIntent->amount,
                'currency' => $paymentIntent->currency,
                'key' => $this->publicKey,
            ];
        } catch (ApiErrorException $e) {
            throw new \Exception('Stripe order creation failed: ' . $e->getMessage());
        }
    }

    public function verifyPayment(array $paymentData): array
    {
        try {
            $paymentIntentId = $paymentData['payment_intent_id'] ?? $paymentData['order_id'];
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);

            if ($paymentIntent->status === 'succeeded') {
                return [
                    'success' => true,
                    'payment_id' => $paymentIntent->id,
                    'order_id' => $paymentIntent->id,
                ];
            }

            return [
                'success' => false,
                'error' => 'Payment not completed. Status: ' . $paymentIntent->status,
            ];
        } catch (ApiErrorException $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getPaymentDetails(string $paymentId): ?array
    {
        try {
            $paymentIntent = PaymentIntent::retrieve($paymentId);

            return [
                'id' => $paymentIntent->id,
                'amount' => $paymentIntent->amount / 100,
                'currency' => strtoupper($paymentIntent->currency),
                'status' => $paymentIntent->status,
                'method' => 'card',
            ];
        } catch (ApiErrorException $e) {
            return null;
        }
    }
}
