<?php

namespace App\Services\PaymentGateway;

use App\Models\SocietySetting;

/**
 * Factory for creating payment gateways based on Society settings.
 * Used for bill payments, facility bookings, and collector payments.
 */
class SocietyPaymentGatewayFactory
{
    /**
     * Get all available payment gateways for a society.
     *
     * @return PaymentGatewayInterface[]
     */
    public static function getAvailableGateways(SocietySetting $settings): array
    {
        $gateways = [];

        // Razorpay
        if ($settings->razorpay_enabled && !empty($settings->razorpay_key_id) && !empty($settings->razorpay_key_secret)) {
            $gateways[] = new RazorpayGateway(
                $settings->razorpay_key_id,
                $settings->razorpay_key_secret,
                true
            );
        }

        // Stripe
        if ($settings->stripe_enabled && !empty($settings->stripe_public_key) && !empty($settings->stripe_secret_key)) {
            $gateways[] = new StripeGateway(
                $settings->stripe_public_key,
                $settings->stripe_secret_key,
                true
            );
        }

        // PayU
        if ($settings->payu_enabled && !empty($settings->payu_merchant_key) && !empty($settings->payu_merchant_salt)) {
            $gateways[] = new PayUGateway(
                $settings->payu_merchant_key,
                $settings->payu_merchant_salt,
                false // Set to true for test mode
            );
        }

        // Cashfree
        if ($settings->cashfree_enabled && !empty($settings->cashfree_app_id) && !empty($settings->cashfree_secret_key)) {
            $gateways[] = new CashfreeGateway(
                $settings->cashfree_app_id,
                $settings->cashfree_secret_key,
                false // Set to true for sandbox mode
            );
        }

        return $gateways;
    }

    /**
     * Get a specific gateway by identifier for a society.
     */
    public static function getGateway(SocietySetting $settings, string $identifier): ?PaymentGatewayInterface
    {
        $gateways = self::getAvailableGateways($settings);

        foreach ($gateways as $gateway) {
            if ($gateway->getIdentifier() === $identifier) {
                return $gateway;
            }
        }

        return null;
    }

    /**
     * Get the default (first available) gateway for a society.
     */
    public static function getDefaultGateway(SocietySetting $settings): ?PaymentGatewayInterface
    {
        $gateways = self::getAvailableGateways($settings);
        return $gateways[0] ?? null;
    }

    /**
     * Get gateway configs for frontend (public keys only).
     */
    public static function getGatewayConfigs(SocietySetting $settings): array
    {
        $gateways = self::getAvailableGateways($settings);
        $configs = [];

        foreach ($gateways as $gateway) {
            $configs[] = [
                'id' => $gateway->getIdentifier(),
                'name' => $gateway->getName(),
                'public_key' => $gateway->getPublicKey(),
            ];
        }

        return $configs;
    }
}
