<?php

namespace App\Services\PaymentGateway;

use Razorpay\Api\Api;
use Razorpay\Api\Errors\SignatureVerificationError;

class RazorpayGateway implements PaymentGatewayInterface
{
    protected string $keyId;
    protected string $keySecret;
    protected bool $enabled;

    public function __construct(string $keyId, string $keySecret, bool $enabled = true)
    {
        $this->keyId = $keyId;
        $this->keySecret = $keySecret;
        $this->enabled = $enabled;
    }

    public function getIdentifier(): string
    {
        return 'razorpay';
    }

    public function getName(): string
    {
        return 'Razorpay';
    }

    public function isEnabled(): bool
    {
        return $this->enabled && !empty($this->keyId) && !empty($this->keySecret);
    }

    public function getPublicKey(): ?string
    {
        return $this->keyId;
    }

    public function createOrder(float $amount, string $currency, array $metadata = []): array
    {
        $api = new Api($this->keyId, $this->keySecret);

        $orderData = [
            'receipt' => $metadata['receipt'] ?? 'order_' . time(),
            'amount' => (int) ($amount * 100), // Convert to paise
            'currency' => $currency,
            'notes' => $metadata['notes'] ?? [],
        ];

        $order = $api->order->create($orderData);

        return [
            'gateway' => $this->getIdentifier(),
            'order_id' => $order->id,
            'amount' => $order->amount,
            'currency' => $order->currency,
            'key' => $this->keyId,
        ];
    }

    public function verifyPayment(array $paymentData): array
    {
        try {
            $api = new Api($this->keyId, $this->keySecret);

            $attributes = [
                'razorpay_order_id' => $paymentData['razorpay_order_id'],
                'razorpay_payment_id' => $paymentData['razorpay_payment_id'],
                'razorpay_signature' => $paymentData['razorpay_signature'],
            ];

            $api->utility->verifyPaymentSignature($attributes);

            return [
                'success' => true,
                'payment_id' => $paymentData['razorpay_payment_id'],
                'order_id' => $paymentData['razorpay_order_id'],
            ];
        } catch (SignatureVerificationError $e) {
            return [
                'success' => false,
                'error' => 'Payment signature verification failed',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getPaymentDetails(string $paymentId): ?array
    {
        try {
            $api = new Api($this->keyId, $this->keySecret);
            $payment = $api->payment->fetch($paymentId);

            return [
                'id' => $payment->id,
                'amount' => $payment->amount / 100,
                'currency' => $payment->currency,
                'status' => $payment->status,
                'method' => $payment->method,
            ];
        } catch (\Exception $e) {
            return null;
        }
    }
}
