<?php

namespace App\Services\PaymentGateway;

use App\Models\SystemSetting;

class PaymentGatewayFactory
{
    /**
     * Get all available payment gateways based on system settings.
     *
     * @return PaymentGatewayInterface[]
     */
    public static function getAvailableGateways(): array
    {
        $settings = SystemSetting::whereIn('key', [
            'razorpay_enabled', 'razorpay_key', 'razorpay_secret',
            'stripe_enabled', 'stripe_key', 'stripe_secret',
            'paypal_enabled', 'paypal_client_id', 'paypal_secret', 'paypal_sandbox',
        ])->pluck('value', 'key')->toArray();

        $gateways = [];

        // Razorpay
        if (($settings['razorpay_enabled'] ?? '0') === '1') {
            $gateway = new RazorpayGateway(
                $settings['razorpay_key'] ?? '',
                $settings['razorpay_secret'] ?? '',
                true
            );
            if ($gateway->isEnabled()) {
                $gateways[] = $gateway;
            }
        }

        // Stripe
        if (($settings['stripe_enabled'] ?? '0') === '1') {
            $gateway = new StripeGateway(
                $settings['stripe_key'] ?? '',
                $settings['stripe_secret'] ?? '',
                true
            );
            if ($gateway->isEnabled()) {
                $gateways[] = $gateway;
            }
        }

        // PayPal
        if (($settings['paypal_enabled'] ?? '0') === '1') {
            $gateway = new PayPalGateway(
                $settings['paypal_client_id'] ?? '',
                $settings['paypal_secret'] ?? '',
                true,
                ($settings['paypal_sandbox'] ?? '1') === '1'
            );
            if ($gateway->isEnabled()) {
                $gateways[] = $gateway;
            }
        }

        return $gateways;
    }

    /**
     * Get a specific payment gateway by identifier.
     */
    public static function getGateway(string $identifier): ?PaymentGatewayInterface
    {
        $gateways = self::getAvailableGateways();

        foreach ($gateways as $gateway) {
            if ($gateway->getIdentifier() === $identifier) {
                return $gateway;
            }
        }

        return null;
    }

    /**
     * Get the default (first available) payment gateway.
     */
    public static function getDefaultGateway(): ?PaymentGatewayInterface
    {
        $gateways = self::getAvailableGateways();
        return $gateways[0] ?? null;
    }

    /**
     * Get gateway configs for frontend (public keys only).
     */
    public static function getGatewayConfigs(): array
    {
        $gateways = self::getAvailableGateways();
        $configs = [];

        foreach ($gateways as $gateway) {
            $configs[] = [
                'id' => $gateway->getIdentifier(),
                'name' => $gateway->getName(),
                'public_key' => $gateway->getPublicKey(),
            ];
        }

        return $configs;
    }
}
