<?php

namespace App\Services\PaymentGateway;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * PayU Payment Gateway Implementation
 * https://developer.payu.in/
 */
class PayUGateway implements PaymentGatewayInterface
{
    protected string $merchantKey;
    protected string $merchantSalt;
    protected bool $testMode;
    protected string $baseUrl;

    public function __construct(string $merchantKey, string $merchantSalt, bool $testMode = false)
    {
        $this->merchantKey = $merchantKey;
        $this->merchantSalt = $merchantSalt;
        $this->testMode = $testMode;
        $this->baseUrl = $testMode 
            ? 'https://test.payu.in' 
            : 'https://secure.payu.in';
    }

    public function createOrder(float $amount, string $currency, array $metadata = []): array
    {
        $txnId = 'TXN' . time() . rand(1000, 9999);
        
        // PayU requires a hash for security
        $hashString = $this->merchantKey . '|' . $txnId . '|' . $amount . '|' . 
                      ($metadata['productinfo'] ?? 'Bill Payment') . '|' . 
                      ($metadata['firstname'] ?? 'Customer') . '|' . 
                      ($metadata['email'] ?? 'customer@example.com') . '|||||||||||' . 
                      $this->merchantSalt;
        
        $hash = strtolower(hash('sha512', $hashString));

        return [
            'order_id' => $txnId,
            'key' => $this->merchantKey,
            'hash' => $hash,
            'amount' => $amount,
            'txnid' => $txnId,
            'productinfo' => $metadata['productinfo'] ?? 'Bill Payment',
            'firstname' => $metadata['firstname'] ?? 'Customer',
            'email' => $metadata['email'] ?? '',
            'phone' => $metadata['phone'] ?? '',
            'surl' => $metadata['success_url'] ?? url('/api/payu/success'),
            'furl' => $metadata['failure_url'] ?? url('/api/payu/failure'),
            'action_url' => $this->baseUrl . '/_payment',
        ];
    }

    public function verifyPayment(array $paymentData): array
    {
        // PayU sends response to callback URL
        // Verify hash from response
        $status = $paymentData['status'] ?? '';
        $txnid = $paymentData['txnid'] ?? '';
        $amount = $paymentData['amount'] ?? '';
        $productinfo = $paymentData['productinfo'] ?? '';
        $firstname = $paymentData['firstname'] ?? '';
        $email = $paymentData['email'] ?? '';
        $mihpayid = $paymentData['mihpayid'] ?? '';
        $receivedHash = $paymentData['hash'] ?? '';

        // Reverse hash calculation
        $hashString = $this->merchantSalt . '|' . $status . '|||||||||||' . 
                      $email . '|' . $firstname . '|' . $productinfo . '|' . 
                      $amount . '|' . $txnid . '|' . $this->merchantKey;
        
        $calculatedHash = strtolower(hash('sha512', $hashString));

        if ($calculatedHash !== $receivedHash) {
            return [
                'success' => false,
                'error' => 'Hash verification failed',
            ];
        }

        if (strtolower($status) === 'success') {
            return [
                'success' => true,
                'payment_id' => $mihpayid,
                'order_id' => $txnid,
            ];
        }

        return [
            'success' => false,
            'error' => 'Payment failed: ' . $status,
        ];
    }

    public function getIdentifier(): string
    {
        return 'payu';
    }

    public function getName(): string
    {
        return 'PayU';
    }

    public function isEnabled(): bool
    {
        return !empty($this->merchantKey) && !empty($this->merchantSalt);
    }

    public function getPublicKey(): ?string
    {
        return $this->merchantKey;
    }

    public function getPaymentDetails(string $paymentId): ?array
    {
        try {
            // PayU doesn't have a direct API to fetch payment details by payment ID
            // You would need to track order IDs and use verification endpoint
            // For now, return null as PayU verification is done via callback
            return null;
        } catch (\Exception $e) {
            Log::error('PayU getPaymentDetails exception', ['error' => $e->getMessage()]);
            return null;
        }
    }
}
