<?php

namespace App\Services\PaymentGateway;

use Illuminate\Support\Facades\Log;
use PaypalServerSdkLib\Authentication\ClientCredentialsAuthCredentialsBuilder;
use PaypalServerSdkLib\Environment;
use PaypalServerSdkLib\Models\Builders\AmountWithBreakdownBuilder;
use PaypalServerSdkLib\Models\Builders\OrderApplicationContextBuilder;
use PaypalServerSdkLib\Models\Builders\OrderRequestBuilder;
use PaypalServerSdkLib\Models\Builders\PurchaseUnitRequestBuilder;
use PaypalServerSdkLib\Models\CheckoutPaymentIntent;
use PaypalServerSdkLib\PaypalServerSdkClient;
use PaypalServerSdkLib\PaypalServerSdkClientBuilder;

class PayPalGateway implements PaymentGatewayInterface
{
    protected string $clientId;
    protected string $clientSecret;
    protected bool $enabled;
    protected bool $sandbox;

    public function __construct(string $clientId, string $clientSecret, bool $enabled = true, bool $sandbox = true)
    {
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        $this->enabled = $enabled;
        $this->sandbox = $sandbox;
    }

    public function getIdentifier(): string
    {
        return 'paypal';
    }

    public function getName(): string
    {
        return 'PayPal';
    }

    public function isEnabled(): bool
    {
        return $this->enabled && !empty($this->clientId) && !empty($this->clientSecret);
    }

    public function getPublicKey(): ?string
    {
        return $this->clientId;
    }

    protected function client(): PaypalServerSdkClient
    {
        return PaypalServerSdkClientBuilder::init()
            ->clientCredentialsAuthCredentials(
                ClientCredentialsAuthCredentialsBuilder::init($this->clientId, $this->clientSecret)
            )
            ->environment($this->sandbox ? Environment::SANDBOX : Environment::PRODUCTION)
            ->build();
    }

    public function createOrder(float $amount, string $currency, array $metadata = []): array
    {
        try {
            $receipt = (string) ($metadata['receipt'] ?? ('order_' . time()));
            $description = (string) ($metadata['description'] ?? 'Payment');

            $orderRequest = OrderRequestBuilder::init(
                CheckoutPaymentIntent::CAPTURE,
                [
                    PurchaseUnitRequestBuilder::init(
                        AmountWithBreakdownBuilder::init(
                            $currency,
                            number_format($amount, 2, '.', '')
                        )->build()
                    )
                        ->referenceId($receipt)
                        ->invoiceId($receipt)
                        ->description($description)
                        ->build(),
                ]
            )
                ->applicationContext(
                    OrderApplicationContextBuilder::init()
                        ->returnUrl($metadata['return_url'] ?? url('/payment/success'))
                        ->cancelUrl($metadata['cancel_url'] ?? url('/payment/cancel'))
                        ->build()
                )
                ->build();

            $client = $this->client();
            $apiResponse = $client->getOrdersController()->createOrder([
                'body' => $orderRequest,
                'prefer' => 'return=representation',
            ]);

            $order = $apiResponse->getResult();
            $orderId = $order?->getId();

            if (!$orderId) {
                throw new \Exception('PayPal order creation failed: missing order id');
            }

            $approvalUrl = null;
            $links = $order->getLinks() ?? [];
            foreach ($links as $link) {
                try {
                    if ($link->getRel() === 'approve') {
                        $approvalUrl = $link->getHref();
                        break;
                    }
                } catch (\Throwable $e) {
                    // Ignore malformed link objects and continue searching.
                }
            }

            return [
                'gateway' => $this->getIdentifier(),
                'order_id' => $orderId,
                'approval_url' => $approvalUrl,
                'amount' => $amount * 100, // Keep consistent with other gateways
                'currency' => $currency,
                'key' => $this->clientId,
            ];
        } catch (\Exception $e) {
            throw new \Exception('PayPal order creation failed: ' . $e->getMessage());
        }
    }

    public function verifyPayment(array $paymentData): array
    {
        try {
            $orderId = $paymentData['order_id']
                ?? $paymentData['paypal_order_id']
                ?? $paymentData['token']
                ?? null;

            if (!$orderId) {
                return [
                    'success' => false,
                    'error' => 'Missing PayPal order id',
                ];
            }

            $client = $this->client();
            $apiResponse = $client->getOrdersController()->captureOrder([
                'id' => $orderId,
                'prefer' => 'return=representation',
            ]);

            $order = $apiResponse->getResult();
            $status = (string) ($order?->getStatus() ?? '');

            $captureId = null;
            $purchaseUnits = $order?->getPurchaseUnits() ?? [];
            $firstPu = $purchaseUnits[0] ?? null;
            $payments = $firstPu?->getPayments();
            $captures = $payments?->getCaptures() ?? [];
            $firstCapture = $captures[0] ?? null;
            if ($firstCapture && method_exists($firstCapture, 'getId')) {
                $captureId = $firstCapture->getId();
            }

            if ($status === 'COMPLETED') {
                return [
                    'success' => true,
                    'payment_id' => is_string($captureId) && $captureId !== '' ? $captureId : (string) $orderId,
                    'order_id' => $orderId,
                ];
            }

            return [
                'success' => false,
                'error' => 'PayPal payment not completed. Status: ' . ($status ?: 'unknown'),
            ];
        } catch (\Throwable $e) {
            Log::warning('PayPal payment verification failed', [
                'error' => $e->getMessage(),
                'order_id' => $paymentData['order_id'] ?? $paymentData['paypal_order_id'] ?? $paymentData['token'] ?? null,
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getPaymentDetails(string $paymentId): ?array
    {
        try {
            $client = $this->client();
            $apiResponse = $client->getOrdersController()->getOrder([
                'id' => $paymentId,
            ]);

            $order = $apiResponse->getResult();
            if (!$order) {
                return null;
            }

            $purchaseUnits = $order->getPurchaseUnits() ?? [];
            $firstPu = $purchaseUnits[0] ?? null;
            $amountObj = $firstPu?->getAmount();
            $amountValue = $amountObj?->getValue();
            $currency = $amountObj?->getCurrencyCode();

            return [
                'id' => (string) ($order->getId() ?? $paymentId),
                'amount' => is_numeric($amountValue) ? (float) $amountValue : 0,
                'currency' => is_string($currency) && $currency !== '' ? $currency : 'USD',
                'status' => (string) ($order->getStatus() ?? 'unknown'),
                'method' => 'paypal',
            ];
        } catch (\Exception $e) {
            return null;
        }
    }
}
