<?php

namespace App\Services\PaymentGateway;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Cashfree Payment Gateway Implementation
 * https://docs.cashfree.com/reference/pg-new-apis-endpoint
 */
class CashfreeGateway implements PaymentGatewayInterface
{
    protected string $appId;
    protected string $secretKey;
    protected bool $testMode;
    protected string $baseUrl;

    public function __construct(string $appId, string $secretKey, bool $testMode = false)
    {
        $this->appId = $appId;
        $this->secretKey = $secretKey;
        $this->testMode = $testMode;
        $this->baseUrl = $testMode 
            ? 'https://sandbox.cashfree.com/pg' 
            : 'https://api.cashfree.com/pg';
    }

    public function createOrder(float $amount, string $currency, array $metadata = []): array
    {
        try {
            $orderId = 'order_' . time() . '_' . rand(1000, 9999);

            $response = Http::withHeaders([
                'x-client-id' => $this->appId,
                'x-client-secret' => $this->secretKey,
                'x-api-version' => '2023-08-01',
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/orders', [
                'order_id' => $orderId,
                'order_amount' => $amount,
                'order_currency' => $currency,
                'order_note' => $metadata['notes']['type'] ?? 'Bill Payment',
                'customer_details' => [
                    'customer_id' => $metadata['notes']['user_id'] ?? 'customer_' . time(),
                    'customer_name' => $metadata['customer_name'] ?? 'Customer',
                    'customer_email' => $metadata['customer_email'] ?? 'customer@example.com',
                    'customer_phone' => $metadata['customer_phone'] ?? '9999999999',
                ],
                'order_meta' => [
                    'return_url' => $metadata['return_url'] ?? url('/api/cashfree/callback?order_id={order_id}'),
                    'notify_url' => $metadata['notify_url'] ?? url('/api/cashfree/webhook'),
                ],
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                return [
                    'order_id' => $data['order_id'],
                    'key' => $this->appId,
                    'payment_session_id' => $data['payment_session_id'],
                    'order_token' => $data['order_token'] ?? null,
                    'cf_order_id' => $data['cf_order_id'] ?? null,
                    'order_status' => $data['order_status'],
                ];
            }

            Log::error('Cashfree order creation failed', ['response' => $response->json()]);
            throw new \Exception('Failed to create Cashfree order: ' . ($response->json()['message'] ?? 'Unknown error'));

        } catch (\Exception $e) {
            Log::error('Cashfree order exception', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function verifyPayment(array $paymentData): array
    {
        try {
            $orderId = $paymentData['order_id'] ?? $paymentData['cashfree_order_id'] ?? null;

            if (!$orderId) {
                return [
                    'success' => false,
                    'error' => 'Order ID not provided',
                ];
            }

            // Fetch order status from Cashfree
            $response = Http::withHeaders([
                'x-client-id' => $this->appId,
                'x-client-secret' => $this->secretKey,
                'x-api-version' => '2023-08-01',
            ])->get($this->baseUrl . '/orders/' . $orderId);

            if ($response->successful()) {
                $data = $response->json();
                
                if ($data['order_status'] === 'PAID') {
                    // Get payment details
                    $paymentsResponse = Http::withHeaders([
                        'x-client-id' => $this->appId,
                        'x-client-secret' => $this->secretKey,
                        'x-api-version' => '2023-08-01',
                    ])->get($this->baseUrl . '/orders/' . $orderId . '/payments');

                    $payments = $paymentsResponse->json();
                    $paymentId = $payments[0]['cf_payment_id'] ?? $orderId;

                    return [
                        'success' => true,
                        'payment_id' => $paymentId,
                        'order_id' => $orderId,
                    ];
                }

                return [
                    'success' => false,
                    'error' => 'Payment not completed. Status: ' . $data['order_status'],
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to verify payment',
            ];

        } catch (\Exception $e) {
            Log::error('Cashfree verification exception', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getIdentifier(): string
    {
        return 'cashfree';
    }

    public function getName(): string
    {
        return 'Cashfree';
    }

    public function isEnabled(): bool
    {
        return !empty($this->appId) && !empty($this->secretKey);
    }

    public function getPublicKey(): ?string
    {
        return $this->appId;
    }

    public function getPaymentDetails(string $paymentId): ?array
    {
        try {
            // Fetch payment details from Cashfree API
            $response = Http::withHeaders([
                'x-client-id' => $this->appId,
                'x-client-secret' => $this->secretKey,
                'x-api-version' => '2023-08-01',
            ])->get($this->baseUrl . '/payments/' . $paymentId);

            if ($response->successful()) {
                $data = $response->json();
                
                return [
                    'id' => $data['cf_payment_id'] ?? $paymentId,
                    'amount' => $data['payment_amount'] ?? 0,
                    'currency' => $data['payment_currency'] ?? 'INR',
                    'status' => $data['payment_status'] ?? 'unknown',
                    'method' => $data['payment_method'] ?? 'unknown',
                ];
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Cashfree getPaymentDetails exception', ['error' => $e->getMessage()]);
            return null;
        }
    }
}
