<?php

namespace App\Services\Messaging;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Twilio WhatsApp Gateway
 * https://www.twilio.com/docs/whatsapp/api
 */
class TwilioWhatsAppGateway implements WhatsAppGatewayInterface
{
    protected string $accountSid;
    protected string $authToken;
    protected string $fromNumber;

    public function __construct(string $accountSid, string $authToken, string $fromNumber)
    {
        $this->accountSid = $accountSid;
        $this->authToken = $authToken;
        // Twilio WhatsApp requires 'whatsapp:' prefix
        $this->fromNumber = str_starts_with($fromNumber, 'whatsapp:') 
            ? $fromNumber 
            : 'whatsapp:' . $fromNumber;
    }

    public function send(string $phone, string $message): bool
    {
        try {
            // Clean and format phone
            $phone = ltrim($phone, '+');
            if (strlen($phone) === 10) {
                $phone = '91' . $phone;
            }
            $phone = 'whatsapp:+' . $phone;

            $url = "https://api.twilio.com/2010-04-01/Accounts/{$this->accountSid}/Messages.json";

            $response = Http::withBasicAuth($this->accountSid, $this->authToken)
                ->asForm()
                ->post($url, [
                    'From' => $this->fromNumber,
                    'To' => $phone,
                    'Body' => $message,
                ]);

            if ($response->successful()) {
                return true;
            }

            Log::error('Twilio WhatsApp failed', ['phone' => $phone, 'error' => $response->json()]);
            return false;

        } catch (\Exception $e) {
            Log::error('Twilio WhatsApp exception', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function sendTemplate(string $phone, string $templateName, array $params = []): bool
    {
        // Twilio uses Content Templates differently
        // For now, use regular message
        $message = $params['body'] ?? "Notification from Societify";
        return $this->send($phone, $message);
    }

    public function getIdentifier(): string
    {
        return 'twilio';
    }
}
