<?php

namespace App\Services\Messaging;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Textlocal SMS Gateway
 * https://www.textlocal.in/docs/api/sms-api
 */
class TextlocalSmsGateway implements SmsGatewayInterface
{
    protected string $apiKey;
    protected string $sender;

    public function __construct(string $apiKey, string $sender)
    {
        $this->apiKey = $apiKey;
        $this->sender = $sender;
    }

    public function send(string $phone, string $message): bool
    {
        try {
            // Clean phone number
            $phone = ltrim($phone, '+');
            
            // Add country code if not present
            if (strlen($phone) === 10) {
                $phone = '91' . $phone;
            }

            $url = 'https://api.textlocal.in/send/';

            $response = Http::asForm()->post($url, [
                'apikey' => $this->apiKey,
                'numbers' => $phone,
                'message' => $message,
                'sender' => $this->sender,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                if (isset($data['status']) && $data['status'] === 'success') {
                    // SMS sent successfully
                    return true;
                }
            }

            Log::error('Textlocal SMS failed', ['phone' => $phone, 'response' => $response->json()]);
            return false;

        } catch (\Exception $e) {
            Log::error('Textlocal SMS exception', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function getIdentifier(): string
    {
        return 'textlocal';
    }
}
