<?php

namespace App\Services\Messaging;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * MSG91 SMS Gateway
 * https://docs.msg91.com/reference/send-sms
 */
class Msg91SmsGateway implements SmsGatewayInterface
{
    protected string $authKey;
    protected string $senderId;
    protected string $route;

    public function __construct(string $authKey, string $senderId, string $route = '4')
    {
        $this->authKey = $authKey;
        $this->senderId = $senderId;
        $this->route = $route; // 4 = transactional, 1 = promotional
    }

    public function send(string $phone, string $message): bool
    {
        try {
            // Clean phone number (remove + if present)
            $phone = ltrim($phone, '+');
            
            // Add country code if not present
            if (strlen($phone) === 10) {
                $phone = '91' . $phone;
            }

            $url = 'https://api.msg91.com/api/v5/flow/';

            // For simple SMS (non-template based)
            $simpleUrl = 'https://api.msg91.com/api/sendhttp.php';
            
            $response = Http::get($simpleUrl, [
                'authkey' => $this->authKey,
                'mobiles' => $phone,
                'message' => $message,
                'sender' => $this->senderId,
                'route' => $this->route,
                'country' => '91',
            ]);

            if ($response->successful()) {
                $body = $response->body();
                // MSG91 returns message ID on success
                if (!str_contains($body, 'Error')) {
                    // SMS sent successfully
                    return true;
                }
            }

            Log::error('MSG91 SMS failed', ['phone' => $phone, 'response' => $response->body()]);
            return false;

        } catch (\Exception $e) {
            Log::error('MSG91 SMS exception', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function getIdentifier(): string
    {
        return 'msg91';
    }
}
