<?php

namespace App\Services\Messaging;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Meta (Facebook) WhatsApp Business API Gateway
 * https://developers.facebook.com/docs/whatsapp/cloud-api/reference/messages
 */
class MetaWhatsAppGateway implements WhatsAppGatewayInterface
{
    protected string $accessToken;
    protected string $phoneNumberId;
    protected string $version;

    public function __construct(string $accessToken, string $phoneNumberId, string $version = 'v18.0')
    {
        $this->accessToken = $accessToken;
        $this->phoneNumberId = $phoneNumberId;
        $this->version = $version;
    }

    public function send(string $phone, string $message): bool
    {
        try {
            // Clean phone number
            $phone = ltrim($phone, '+');
            if (strlen($phone) === 10) {
                $phone = '91' . $phone;
            }

            $url = "https://graph.facebook.com/{$this->version}/{$this->phoneNumberId}/messages";

            $response = Http::withToken($this->accessToken)
                ->post($url, [
                    'messaging_product' => 'whatsapp',
                    'recipient_type' => 'individual',
                    'to' => $phone,
                    'type' => 'text',
                    'text' => [
                        'preview_url' => false,
                        'body' => $message,
                    ],
                ]);

            if ($response->successful()) {
                return true;
            }

            Log::error('Meta WhatsApp failed', ['phone' => $phone, 'error' => $response->json()]);
            return false;

        } catch (\Exception $e) {
            Log::error('Meta WhatsApp exception', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function sendTemplate(string $phone, string $templateName, array $params = []): bool
    {
        try {
            $phone = ltrim($phone, '+');
            if (strlen($phone) === 10) {
                $phone = '91' . $phone;
            }

            $url = "https://graph.facebook.com/{$this->version}/{$this->phoneNumberId}/messages";

            $templateData = [
                'messaging_product' => 'whatsapp',
                'recipient_type' => 'individual',
                'to' => $phone,
                'type' => 'template',
                'template' => [
                    'name' => $templateName,
                    'language' => [
                        'code' => $params['language'] ?? 'en',
                    ],
                ],
            ];

            // Add template components if provided
            if (!empty($params['components'])) {
                $templateData['template']['components'] = $params['components'];
            }

            $response = Http::withToken($this->accessToken)->post($url, $templateData);

            if ($response->successful()) {
                return true;
            }

            Log::error('Meta WhatsApp template failed', ['phone' => $phone, 'error' => $response->json()]);
            return false;

        } catch (\Exception $e) {
            Log::error('Meta WhatsApp template exception', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function getIdentifier(): string
    {
        return 'meta';
    }
}
