<?php

namespace App\Services\Messaging;

use App\Models\SocietySetting;
use Illuminate\Support\Facades\Log;

/**
 * Factory for creating SMS and WhatsApp gateways based on society settings.
 */
class MessagingGatewayFactory
{
    /**
     * Get SMS gateway based on society settings.
     */
    public static function getSmsGateway(SocietySetting $settings): ?SmsGatewayInterface
    {
        if (!$settings->sms_enabled || empty($settings->sms_api_key)) {
            return null;
        }

        $provider = strtolower($settings->sms_provider ?? 'msg91');

        return match ($provider) {
            'twilio' => new TwilioSmsGateway(
                $settings->sms_username,      // Account SID
                $settings->sms_api_key,       // Auth Token
                $settings->sms_sender_id      // From Number
            ),
            'msg91' => new Msg91SmsGateway(
                $settings->sms_api_key,
                $settings->sms_sender_id
            ),
            'textlocal' => new TextlocalSmsGateway(
                $settings->sms_api_key,
                $settings->sms_sender_id
            ),
            default => new Msg91SmsGateway(
                $settings->sms_api_key,
                $settings->sms_sender_id
            ),
        };
    }

    /**
     * Get WhatsApp gateway based on society settings.
     */
    public static function getWhatsAppGateway(SocietySetting $settings): ?WhatsAppGatewayInterface
    {
        if (!$settings->whatsapp_enabled || empty($settings->whatsapp_api_key)) {
            return null;
        }

        $provider = strtolower($settings->whatsapp_provider ?? 'meta');

        return match ($provider) {
            'twilio' => new TwilioWhatsAppGateway(
                $settings->whatsapp_business_id,  // Account SID
                $settings->whatsapp_api_key,      // Auth Token
                $settings->whatsapp_sender_id     // From Number
            ),
            'meta', 'facebook' => new MetaWhatsAppGateway(
                $settings->whatsapp_api_key,      // Access Token
                $settings->whatsapp_business_id   // Phone Number ID
            ),
            default => new MetaWhatsAppGateway(
                $settings->whatsapp_api_key,
                $settings->whatsapp_business_id
            ),
        };
    }

    /**
     * Check if SMS is properly configured.
     */
    public static function isSmsConfigured(SocietySetting $settings): bool
    {
        return $settings->sms_enabled 
            && !empty($settings->sms_api_key)
            && !empty($settings->sms_sender_id);
    }

    /**
     * Check if WhatsApp is properly configured.
     */
    public static function isWhatsAppConfigured(SocietySetting $settings): bool
    {
        return $settings->whatsapp_enabled 
            && !empty($settings->whatsapp_api_key);
    }
}
