<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;

class FirebaseNotificationService
{
    protected $messaging = null;
    protected bool $enabled = false;

    public function __construct()
    {
        $this->initializeFirebase();
    }

    /**
     * Initialize Firebase with credentials from config
     */
    protected function initializeFirebase(): void
    {
        try {
            $credentialsPath = config('services.firebase.credentials');

            if (!$credentialsPath || !file_exists($credentialsPath)) {
                Log::warning('Firebase credentials not found. Push notifications disabled.');
                return;
            }

            $factory = (new Factory)->withServiceAccount($credentialsPath);
            $this->messaging = $factory->createMessaging();
            $this->enabled = true;

            // Firebase initialized successfully
        } catch (\Exception $e) {
            Log::error('Firebase initialization failed: ' . $e->getMessage());
            $this->enabled = false;
        }
    }

    /**
     * Check if Firebase is enabled
     */
    public function isEnabled(): bool
    {
        return $this->enabled && $this->messaging !== null;
    }

    /**
     * Send notification to a user
     */
    public function sendToUser(User $user, string $title, string $body, array $data = []): bool
    {
        if (!$this->isEnabled()) {
            Log::warning('Firebase not enabled. Notification not sent.');
            return false;
        }

        // Skip notification if user has no FCM token
        if (empty($user->fcm_token)) {
            return false;
        }

        return $this->sendToToken($user->fcm_token, $title, $body, $data);
    }

    /**
     * Send notification to a specific FCM token
     */
    public function sendToToken(string $token, string $title, string $body, array $data = []): bool
    {
        if (!$this->isEnabled()) {
            Log::warning('Firebase not enabled. Notification not sent.');
            return false;
        }

        try {
            $notification = Notification::create($title, $body);

            $message = CloudMessage::withTarget('token', $token)
                ->withNotification($notification)
                ->withData($data);

            $this->messaging->send($message);

            // Notification sent successfully
            return true;
        } catch (\Kreait\Firebase\Exception\Messaging\NotFound $e) {
            // Token is invalid or expired
            Log::warning("FCM token invalid or expired: " . $e->getMessage());
            return false;
        } catch (\Exception $e) {
            Log::error("Failed to send notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to multiple users
     */
    public function sendToUsers(array $users, string $title, string $body, array $data = []): array
    {
        $results = [
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
        ];

        foreach ($users as $user) {
            if (empty($user->fcm_token)) {
                $results['skipped']++;
                continue;
            }

            if ($this->sendToUser($user, $title, $body, $data)) {
                $results['success']++;
            } else {
                $results['failed']++;
            }
        }

        return $results;
    }

    /**
     * Send visitor arrival notification to resident
     */
    public function sendVisitorArrivalNotification(User $resident, array $visitorData): bool
    {
        $otp = $visitorData['gatepass_code'] ?? '----';
        $title = 'Visitor Arrived';
        $body = "{$visitorData['name']} is at the gate. OTP: {$otp}. Please approve or share this OTP.";

        $data = [
            'type' => 'visitor',
            'id' => (string) $visitorData['id'],
            'visitor_name' => $visitorData['name'],
            'visitor_phone' => $visitorData['phone'] ?? '',
            'visitor_type' => $visitorData['type'] ?? 'guest',
            'gatepass_code' => $otp,
            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
        ];

        return $this->sendToUser($resident, $title, $body, $data);
    }

    /**
     * Send emergency panic alert
     */
    public function sendEmergencyNotification(array $users, array $alertData): array
    {
        $title = 'EMERGENCY ALERT!';
        $body = $alertData['body'];

        $data = [
            'type' => 'emergency',
            'id' => (string) $alertData['id'],
            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
            'sound' => 'emergency_siren', // For data-only handling fallback
        ];

        // Custom notification with sound
        // Note: For custom sounds to work, the sound file must exist in the app bundle
        $notification = Notification::fromArray([
            'title' => $title,
            'body' => $body,
        ]);
        
        // We need to construct messages manually to add platform specific configs for priority and sound
        $results = [
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
        ];

        if (!$this->isEnabled()) {
            return $results;
        }

        foreach ($users as $user) {
            if (empty($user->fcm_token)) {
                $results['skipped']++;
                continue;
            }

            try {
                $message = CloudMessage::withTarget('token', $user->fcm_token)
                    ->withNotification($notification)
                    ->withData($data)
                    ->withAndroidConfig(\Kreait\Firebase\Messaging\AndroidConfig::fromArray([
                        'priority' => 'high',
                        'notification' => [
                            'sound' => 'emergency_siren',
                            'channel_id' => 'emergency_channel', // App should create this channel
                            'default_sound' => false,
                            'visibility' => 'public',
                        ],
                    ]))
                    ->withApnsConfig(\Kreait\Firebase\Messaging\ApnsConfig::fromArray([
                        'headers' => [
                            'apns-priority' => '10', // High priority
                        ],
                        'payload' => [
                            'aps' => [
                                'sound' => 'emergency_siren.caf',
                                'content-available' => 1,
                            ],
                        ],
                    ]));

                $this->messaging->send($message);
                $results['success']++;
            } catch (\Exception $e) {
                Log::error("Failed to send emergency notification to user {$user->id}: " . $e->getMessage());
                $results['failed']++;
            }
        }

        return $results;
    }

    /**
     * Send visitor approved notification
     */
    public function sendVisitorApprovedNotification(string $guardToken, array $visitorData): bool
    {
        $title = 'Visitor Approved';
        $body = "{$visitorData['name']} has been approved by the resident";

        $data = [
            'type' => 'visitor_approved',
            'id' => (string) $visitorData['id'],
            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
        ];

        return $this->sendToToken($guardToken, $title, $body, $data);
    }

    /**
     * Send visitor rejected notification
     */
    public function sendVisitorRejectedNotification(string $guardToken, array $visitorData): bool
    {
        $title = 'Visitor Rejected';
        $body = "{$visitorData['name']} has been rejected by the resident";

        $data = [
            'type' => 'visitor_rejected',
            'id' => (string) $visitorData['id'],
            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
        ];

        return $this->sendToToken($guardToken, $title, $body, $data);
    }
}
