<?php

namespace App\Services;

use App\Mail\Society\WelcomeSociety;
use App\Mail\Subscription\PaymentReceived;
use App\Mail\Subscription\SubscriptionActivated;
use App\Mail\Subscription\SubscriptionExpired;
use App\Mail\Subscription\SubscriptionExpiringSoon;
use App\Models\Society;
use App\Models\Subscription;
use App\Models\SubscriptionPayment;
use App\Models\SystemSetting;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class EmailService
{
    /**
     * Check if email is enabled in system settings.
     */
    public static function isEmailEnabled(): bool
    {
        $host = SystemSetting::getValue('smtp_host');
        $fromAddress = SystemSetting::getValue('mail_from_address');

        return !empty($host) && !empty($fromAddress);
    }

    /**
     * Get recipients for society emails (society email + admin email).
     */
    public static function getSocietyRecipients(Society $society): array
    {
        $recipients = [];

        // Society email
        if (!empty($society->email)) {
            $recipients[] = $society->email;
        }

        // Society admin email
        $admin = User::where('society_id', $society->id)
            ->whereHas('roles', fn($q) => $q->where('name', 'society_admin'))
            ->first();

        if ($admin && !empty($admin->email) && !in_array($admin->email, $recipients)) {
            $recipients[] = $admin->email;
        }

        return $recipients;
    }

    /**
     * Send welcome email to new society.
     */
    public static function sendWelcomeEmail(Society $society, User $admin, ?string $plainPassword = null): void
    {
        // Skip if email not configured
        if (!self::isEmailEnabled()) {
            return;
        }

        try {
            $recipients = self::getSocietyRecipients($society);
            
            if (empty($recipients)) {
                $recipients = [$admin->email];
            }

            foreach ($recipients as $recipient) {
                Mail::to($recipient)->queue(new WelcomeSociety($society, $admin, $plainPassword));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send welcome email', ['error' => $e->getMessage(), 'society_id' => $society->id]);
        }
    }

    /**
     * Send subscription activated email.
     */
    public static function sendSubscriptionActivatedEmail(Society $society, Subscription $subscription): void
    {
        if (!self::isEmailEnabled()) {
            return;
        }

        try {
            $recipients = self::getSocietyRecipients($society);

            foreach ($recipients as $recipient) {
                Mail::to($recipient)->queue(new SubscriptionActivated($society, $subscription));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send subscription activated email', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Send payment received email.
     */
    public static function sendPaymentReceivedEmail(Society $society, Subscription $subscription, SubscriptionPayment $payment): void
    {
        if (!self::isEmailEnabled()) {
            return;
        }

        try {
            $recipients = self::getSocietyRecipients($society);

            foreach ($recipients as $recipient) {
                Mail::to($recipient)->queue(new PaymentReceived($society, $subscription, $payment));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send payment received email', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Send subscription expiring soon email.
     */
    public static function sendExpiringEmail(Society $society, Subscription $subscription, int $daysRemaining): void
    {
        if (!self::isEmailEnabled()) {
            return;
        }

        try {
            $recipients = self::getSocietyRecipients($society);

            foreach ($recipients as $recipient) {
                Mail::to($recipient)->queue(new SubscriptionExpiringSoon($society, $subscription, $daysRemaining));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send expiring email', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Send subscription expired email.
     */
    public static function sendExpiredEmail(Society $society, Subscription $subscription): void
    {
        if (!self::isEmailEnabled()) {
            return;
        }

        try {
            $recipients = self::getSocietyRecipients($society);

            foreach ($recipients as $recipient) {
                Mail::to($recipient)->queue(new SubscriptionExpired($society, $subscription));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send expired email', ['error' => $e->getMessage()]);
        }
    }
}
