<?php

namespace App\Services;

use App\Models\Bill;
use App\Models\Payment;
use App\Models\Society;
use App\Models\SocietySetting;
use App\Models\User;
use Illuminate\Support\Facades\Log;

/**
 * Unified notification service that broadcasts to all enabled channels
 * based on society settings and notification triggers.
 */
class BroadcastNotificationService
{
    protected SocietySetting $settings;
    protected Society $society;

    public function __construct(Society $society)
    {
        $this->society = $society;
        $this->settings = $society->settings ?? new SocietySetting();
    }

    /**
     * Send bill generated notification.
     */
    public function sendBillGeneratedNotification(Bill $bill, User $resident): void
    {
        if (!$this->settings->notify_bill_generated) {
            return;
        }

        $title = 'New Bill Generated';
        $message = "A new bill #{$bill->bill_no} of ₹{$bill->amount} has been generated. Due: {$bill->due_date->format('d M, Y')}";

        $this->broadcast($resident, $title, $message, [
            'type' => 'bill_generated',
            'bill_id' => (string) $bill->id,
        ]);

        // Also send email
        ResidentEmailService::sendBillGeneratedEmail($bill);
    }

    /**
     * Send payment received notification.
     */
    public function sendPaymentReceivedNotification(Payment $payment, User $resident): void
    {
        if (!$this->settings->notify_payment_received) {
            return;
        }

        $title = 'Payment Received';
        $message = "Your payment of ₹{$payment->amount} has been received successfully.";

        $this->broadcast($resident, $title, $message, [
            'type' => 'payment_received',
            'payment_id' => (string) $payment->id,
        ]);

        // Also send email
        ResidentEmailService::sendPaymentConfirmationEmail($payment);
    }

    /**
     * Send visitor arrival notification.
     */
    public function sendVisitorArrivalNotification(User $resident, array $visitorData): void
    {
        if (!$this->settings->notify_visitor_arrival) {
            return;
        }

        $title = 'Visitor at Gate';
        $message = "{$visitorData['name']} is waiting at the gate for approval.";

        $this->broadcast($resident, $title, $message, [
            'type' => 'visitor_arrival',
            'visitor_id' => (string) ($visitorData['id'] ?? null),
        ]);
    }

    /**
     * Send maintenance request notification.
     */
    public function sendMaintenanceRequestNotification(User $resident, array $ticketData): void
    {
        if (!$this->settings->notify_maintenance_request) {
            return;
        }

        $title = 'Ticket Update';
        $message = $ticketData['message'] ?? "Your maintenance ticket has been updated.";

        $this->broadcast($resident, $title, $message, [
            'type' => 'maintenance_request',
            'ticket_id' => (string) ($ticketData['id'] ?? null),
        ]);
    }

    /**
     * Send announcement notification.
     */
    public function sendAnnouncementNotification(array $users, array $announcementData): void
    {
        if (!$this->settings->notify_announcement) {
            return;
        }

        $title = $announcementData['title'] ?? 'New Announcement';
        $message = $announcementData['message'] ?? 'A new announcement has been posted.';

        foreach ($users as $user) {
            $this->broadcast($user, $title, $message, [
                'type' => 'announcement',
                'announcement_id' => (string) ($announcementData['id'] ?? null),
            ]);
        }
    }

    /**
     * Broadcast message to all enabled channels for a user.
     */
    protected function broadcast(User $user, string $title, string $message, array $data = []): void
    {
        // 1. Push Notification (FCM)
        if ($this->settings->fcm_enabled && $user->fcm_token) {
            $this->sendFcm($user, $title, $message, $data);
        }

        // 2. SMS
        if ($this->settings->sms_enabled && $user->phone) {
            $this->sendSms($user->phone, $message);
        }

        // 3. WhatsApp
        if ($this->settings->whatsapp_enabled && $user->phone) {
            $this->sendWhatsApp($user->phone, $message);
        }
    }

    /**
     * Send FCM push notification.
     */
    protected function sendFcm(User $user, string $title, string $message, array $data = []): void
    {
        try {
            $firebaseService = app(FirebaseNotificationService::class);
            $firebaseService->sendToUser($user, $title, $message, $data);
        } catch (\Exception $e) {
            Log::error('FCM notification failed', ['error' => $e->getMessage(), 'user_id' => $user->id]);
        }
    }

    /**
     * Send SMS notification using configured gateway.
     */
    protected function sendSms(string $phone, string $message): void
    {
        try {
            $gateway = Messaging\MessagingGatewayFactory::getSmsGateway($this->settings);
            
            if (!$gateway) {
                // SMS gateway not configured - skip SMS notification
                return;
            }

            $result = $gateway->send($phone, $message);
            
            if ($result) {
                // SMS sent successfully via gateway
            }
        } catch (\Exception $e) {
            Log::error('SMS notification failed', ['error' => $e->getMessage(), 'phone' => $phone]);
        }
    }

    /**
     * Send WhatsApp notification using configured gateway.
     */
    protected function sendWhatsApp(string $phone, string $message): void
    {
        try {
            $gateway = Messaging\MessagingGatewayFactory::getWhatsAppGateway($this->settings);
            
            if (!$gateway) {
                // WhatsApp gateway not configured - skip WhatsApp notification
                return;
            }

            $result = $gateway->send($phone, $message);
            
            if ($result) {
                // WhatsApp sent successfully via gateway
            }
        } catch (\Exception $e) {
            Log::error('WhatsApp notification failed', ['error' => $e->getMessage(), 'phone' => $phone]);
        }
    }

    /**
     * Static factory for quick access.
     */
    public static function forSociety(Society $society): self
    {
        return new self($society);
    }
}
