<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Visitor;
use Illuminate\Auth\Access\Response;

class VisitorPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->role?->name === 'super_admin') {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermission('visitors.view');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Visitor $model): bool
    {
        // Users can view their own visitors
        if ($user->id === $model->user_id) {
            return true;
        }

        return $user->hasPermission('visitors.view') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasPermission('visitors.create');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Visitor $model): bool
    {
        // Update usually means editing details.
        // If no specific edit permission, maybe not allowed?
        // Or maybe 'visitors.create' allows editing pending visitors?
        // I'll return false for now unless I see 'visitors.edit'.
        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Visitor $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Visitor $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Visitor $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can check in the visitor.
     */
    public function checkIn(User $user, Visitor $model): bool
    {
        return $user->hasPermission('visitors.check_in') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can check out the visitor.
     */
    public function checkOut(User $user, Visitor $model): bool
    {
        if ($user->hasRole('guard') && $user->society_id === $model->society_id) {
            return true;
        }
        return $user->hasPermission('visitors.check_out') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can approve the visitor.
     */
    public function approve(User $user, Visitor $model): bool
    {
        return $user->hasPermission('visitors.approve') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can reject the visitor.
     */
    public function reject(User $user, Visitor $model): bool
    {
        return $user->hasPermission('visitors.reject') && $user->society_id === $model->society_id;
    }
}
