<?php

namespace App\Policies;

use App\Models\Payment;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class PaymentPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->role?->name === 'super_admin') {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermission('payments.view');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Payment $model): bool
    {
        // Users can view their own payments
        if ($user->id === $model->user_id) {
            return true;
        }

        return $user->hasPermission('payments.view') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        // Residents can create payments (pay bills)
        // But usually they do it via specific flow.
        // If 'payments.create' is for admin recording payments, then check permission.
        // Residents might not have 'payments.create' permission in seeder?
        // Let's check seeder.
        // 'payments.create' is in 'payments' group.
        // If resident role has it, they can.
        // If not, maybe they use a different route or we allow them if it's their own payment?
        // But usually `create` policy is for "can I create A payment".
        // I'll check permission.
        return $user->hasPermission('payments.create');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Payment $model): bool
    {
        return $user->hasPermission('payments.reconcile') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Payment $model): bool
    {
        return false; // Payments usually shouldn't be deleted, maybe voided?
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Payment $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Payment $model): bool
    {
        return false;
    }
}
