<?php

namespace App\Policies;

use App\Models\MaintenanceTicket;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class MaintenanceTicketPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->role?->name === 'super_admin') {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermission('maintenance.view');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, MaintenanceTicket $model): bool
    {
        // Users can view their own tickets
        if ($user->id === $model->user_id) {
            return true;
        }

        return $user->hasPermission('maintenance.view') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasPermission('maintenance.create');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, MaintenanceTicket $model): bool
    {
        // Owner can update if open
        if ($user->id === $model->user_id && $model->status === 'open') {
            return true;
        }

        // Vendor can update tickets assigned to them
        if ($user->role?->name === 'vendor' && $user->id === $model->vendor_id && $user->society_id === $model->society_id) {
            return true;
        }

        return $user->hasPermission('maintenance.update_status') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, MaintenanceTicket $model): bool
    {
        // Owner can delete if open
        if ($user->id === $model->user_id && $model->status === 'open') {
            return true;
        }

        return false; // No delete permission for admins in list?
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, MaintenanceTicket $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, MaintenanceTicket $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can assign the ticket.
     */
    public function assign(User $user, MaintenanceTicket $model): bool
    {
        return $user->hasPermission('maintenance.assign') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can resolve the ticket.
     */
    public function resolve(User $user, MaintenanceTicket $model): bool
    {
        return $user->hasPermission('maintenance.resolve') && $user->society_id === $model->society_id;
    }
}
