<?php

namespace App\Policies;

use App\Models\Bill;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class BillPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->role?->name === 'super_admin') {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermission('billing.bills.view');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Bill $model): bool
    {
        // Users can view their own bills
        if ($user->id === $model->user_id) {
            return true;
        }

        // Family members can view bills of their parent/owner
        if ($user->type === 'family_member' && $user->parent_id && $user->parent_id === $model->user_id) {
            return true;
        }

        return $user->hasPermission('billing.bills.view') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasPermission('billing.bills.create');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Bill $model): bool
    {
        if ($user->society_id !== $model->society_id) {
            return false;
        }

        if ($model->status === 'draft') {
            return $user->hasPermission('billing.bills.create');
        }

        return $user->hasPermission('billing.bills.revise');
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Bill $model): bool
    {
        if ($user->society_id !== $model->society_id) {
            return false;
        }

        // Only drafts can be deleted
        if ($model->status === 'draft') {
            return $user->hasPermission('billing.bills.create');
        }

        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Bill $model): bool
    {
        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Bill $model): bool
    {
        return false;
    }
}
