<?php

namespace App\Policies;

use App\Models\Announcement;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class AnnouncementPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->role?->name === 'super_admin') {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermission('announcements.view');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Announcement $model): bool
    {
        // All users in society can view published announcements
        if ($user->society_id === $model->society_id && $model->is_published) {
            return true;
        }

        return $user->hasPermission('announcements.view') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasPermission('announcements.create');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Announcement $model): bool
    {
        return $user->hasPermission('announcements.edit') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Announcement $model): bool
    {
        return $user->hasPermission('announcements.delete') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Announcement $model): bool
    {
        return $user->hasPermission('announcements.create') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Announcement $model): bool
    {
        return $user->hasPermission('announcements.delete') && $user->society_id === $model->society_id;
    }

    /**
     * Determine whether the user can publish the announcement.
     */
    public function publish(User $user, Announcement $model): bool
    {
        return $user->hasPermission('announcements.publish') && $user->society_id === $model->society_id;
    }
}
