<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes, TwoFactorAuthenticatable;

    protected $table = 'users';

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'society_id',
        'role_id',
        'parent_id', // [NEW]
        'type',      // [NEW]
        'name',
        'email',
        'phone',
        'upi_vpa',   // [NEW] UPI Virtual Payment Address
        'password',
        'unit_id',
        'tenant_agreement_end_date', // [NEW]
        'monthly_rent', // [NEW] Monthly rent amount
        'rent_increase_duration_months', // [NEW] Rent increase after X months
        'rent_increase_percentage', // [NEW] Percentage increase
        'fcm_token',
        'status',
        'directory_visible',
        'last_login_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'two_factor_confirmed_at' => 'datetime',
            'last_login_at' => 'datetime',
            'directory_visible' => 'boolean',
            'tenant_agreement_end_date' => 'date',
            'monthly_rent' => 'decimal:2',
            'rent_increase_percentage' => 'decimal:2',
        ];
    }

    // ========== RELATIONSHIPS ==========

    /**
     * Get the parent user (for family members and tenants).
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(User::class, 'parent_id');
    }

    /**
     * Get the family members for the user.
     */
    public function familyMembers(): HasMany
    {
        return $this->hasMany(User::class, 'parent_id')->where('type', 'family_member');
    }

    /**
     * Get the tenants for the user.
     */
    public function tenants(): HasMany
    {
        return $this->hasMany(User::class, 'parent_id')->where('type', 'tenant');
    }

    /**
     * Get rent records for this tenant.
     */
    public function rentRecords(): HasMany
    {
        return $this->hasMany(RentRecord::class, 'tenant_id');
    }

    /**
     * Get rent records for tenants owned by this user (owner).
     */
    public function ownedRentRecords(): HasMany
    {
        return $this->hasMany(RentRecord::class, 'owner_id');
    }

    public function documents(): HasMany
    {
        return $this->hasMany(Document::class);
    }

    /**
     * Get the society that owns the user.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the role that owns the user.
     */
    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class);
    }

    /**
     * Get the unit that owns the user.
     */
    public function unit(): BelongsTo
    {
        return $this->belongsTo(Unit::class);
    }

    /**
     * Get the bills for the user.
     */
    public function bills(): HasMany
    {
        return $this->hasMany(Bill::class);
    }

    /**
     * Get the maintenance tickets for the user.
     */
    public function maintenanceTickets(): HasMany
    {
        return $this->hasMany(MaintenanceTicket::class);
    }

    /**
     * Get the visitors for the user.
     */
    public function visitors(): HasMany
    {
        return $this->hasMany(Visitor::class);
    }

    /**
     * Get the payments for the user.
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the expenses created by the user.
     */
    public function createdExpenses(): HasMany
    {
        return $this->hasMany(Expense::class, 'created_by');
    }

    /**
     * Get the expenses where user is the vendor.
     */
    public function vendorExpenses(): HasMany
    {
        return $this->hasMany(Expense::class, 'vendor_id');
    }

    /**
     * Get the announcements created by the user.
     */
    public function announcements(): HasMany
    {
        return $this->hasMany(Announcement::class, 'created_by');
    }

    /**
     * Get the collector wallet for the user (collectors only).
     */
    public function collectorWallet(): HasOne
    {
        return $this->hasOne(CollectorWallet::class, 'collector_id');
    }

    // ========== SUPER ADMIN SCOPES ==========

    /**
     * Scope: Web Admin Users Only
     */
    public function scopeWebAdmins(Builder $query): Builder
    {
        return $query->whereHas('role', function ($q) {
            $q->whereIn('name', ['society_admin', 'sub_admin', 'accountant']);
        });
    }

    /**
     * Scope: Mobile Users Only
     */
    public function scopeMobileUsers(Builder $query): Builder
    {
        return $query->whereHas('role', function ($q) {
            $q->whereIn('name', ['resident', 'guard', 'vendor', 'staff']);
        });
    }

    /**
     * Scope: By Role Name
     */
    public function scopeRole(Builder $query, string $roleName): Builder
    {
        return $query->whereHas('role', fn ($q) => $q->where('name', $roleName));
    }

    /**
     * Scope: Active Users in Society
     */
    public function scopeActiveInSociety(Builder $query, int $societyId): Builder
    {
        return $query->where('society_id', $societyId)
            ->where('status', 'active');
    }

    /**
     * Scope: Recent Logins (Super Admin Dashboard)
     */
    public function scopeRecentLogins(Builder $query, int $days = 7): Builder
    {
        return $query->whereNotNull('last_login_at')
            ->where('last_login_at', '>=', now()->subDays($days));
    }

    /**
     * Scope: Search by Name/Phone/Email
     */
    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
                ->orWhere('phone', 'LIKE', "%{$search}%")
                ->orWhere('email', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Scope: Directory Visible (Community Feature)
     */
    public function scopeDirectoryVisible(Builder $query): Builder
    {
        return $query->where('directory_visible', true);
    }

    // ========== ROLE METHODS ==========

    /**
     * Check if Web Admin (Society Admin/Sub-Admin/Accountant)
     */
    public function isWebAdmin(): bool
    {
        $webRoles = ['society_admin', 'sub_admin', 'accountant'];

        return in_array($this->role->name ?? '', $webRoles);
    }

    /**
     * Check if Mobile User (Resident/Guard/Vendor/Staff)
     */
    public function isMobileUser(): bool
    {
        $mobileRoles = ['resident', 'guard', 'vendor', 'staff'];

        return in_array($this->role->name ?? '', $mobileRoles);
    }

    /**
     * Check specific role
     */
    public function hasRole(string $roleName): bool
    {
        return ($this->role->name ?? '') === $roleName;
    }

    /**
     * Check if user has a specific permission.
     */
    public function hasPermission(string $permissionName): bool
    {
        if (! $this->role) {
            return false;
        }

        return $this->role->hasPermission($permissionName);
    }

    /**
     * Check if user has any of the given permissions.
     */
    public function hasAnyPermission(array $permissionNames): bool
    {
        if (! $this->role) {
            return false;
        }

        return $this->role->hasAnyPermission($permissionNames);
    }

    /**
     * Update last login timestamp
     */
    public function updateLastLogin(): void
    {
        $this->update(['last_login_at' => now()]);
    }

    // ========== ACCESSORS ==========

    /**
     * Full profile info for directory
     */
    public function getProfileDataAttribute(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'phone' => $this->directory_visible ? $this->phone : '****'.substr($this->phone ?? '', -4),
            'unit_no' => $this->unit?->unit_no ?? 'N/A',
            'role' => $this->role->name ?? 'N/A',
            'society' => $this->society?->name ?? 'N/A',
        ];
    }

    /**
     * Role display name
     */
    public function getRoleDisplayAttribute(): string
    {
        return match ($this->role->name ?? '') {
            'society_admin' => 'Society Admin',
            'sub_admin' => 'Sub Admin',
            'accountant' => 'Accountant',
            'resident' => 'Resident',
            'guard' => 'Security Guard',
            'vendor' => 'Service Vendor',
            'staff' => 'Maintenance Staff',
            default => ucfirst($this->role->name ?? 'Unknown'),
        };
    }

    /**
     * Check if user is an owner.
     */
    public function isOwner(): bool
    {
        return $this->type === 'owner';
    }

    /**
     * Check if user is a tenant.
     */
    public function isTenant(): bool
    {
        return $this->type === 'tenant';
    }

    /**
     * Check if user is a family member.
     */
    public function isFamily(): bool
    {
        return $this->type === 'family_member';
    }
}
