<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Transaction extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'bank_account_id',
        'type', // credit, debit
        'amount',
        'category', // bill_payment, facility_booking, ad_revenue, expense, etc.
        'reference_type',
        'reference_id',
        'description',
        'transaction_date',
        'payment_method',
        'created_by',
        'balance_after'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'transaction_date' => 'date',
    ];

    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function reference(): MorphTo
    {
        return $this->morphTo();
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeCredits($query)
    {
        return $query->where('type', 'credit');
    }

    public function scopeDebits($query)
    {
        return $query->where('type', 'debit');
    }

    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('transaction_date', [$from, $to]);
    }

    /**
     * Get the formatted reference.
     */
    public function getFormattedReferenceAttribute()
    {
        $reference = $this->reference;

        if (! $reference) {
            return '-';
        }

        if ($reference instanceof \App\Models\Payment) {
            return 'Payment #' . $reference->payment_no;
        }

        if ($reference instanceof \App\Models\Expense) {
            return 'Expense #' . $reference->expense_no;
        }

        if ($reference instanceof \App\Models\Bill) {
            return 'Bill #' . $reference->bill_no;
        }
        
        if ($reference instanceof \App\Models\FacilityBooking) {
            return 'Booking #' . $reference->id; 
        }

        if ($reference instanceof \App\Models\Advertisement) {
            return 'Ad: ' . $reference->title;
        }

        return class_basename($reference) . ' #' . $reference->id;
    }
}
