<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class SubscriptionPlan extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'display_name',
        'description',
        'monthly_price',
        'quarterly_price',
        'annual_price',
        'max_users',
        'max_units',
        'max_buildings',
        'storage_gb',
        'sms_enabled',
        'whatsapp_enabled',
        'email_enabled',
        'fcm_enabled',
        'advanced_reports',
        'api_access',
        'custom_branding',
        'priority_support',
        'white_label',
        'is_active',
        'sort_order',
        'validity_days',
    ];

    protected function casts(): array
    {
        return [
            'monthly_price' => 'decimal:2',
            'quarterly_price' => 'decimal:2',
            'annual_price' => 'decimal:2',
            'sms_enabled' => 'boolean',
            'whatsapp_enabled' => 'boolean',
            'email_enabled' => 'boolean',
            'fcm_enabled' => 'boolean',
            'advanced_reports' => 'boolean',
            'api_access' => 'boolean',
            'custom_branding' => 'boolean',
            'priority_support' => 'boolean',
            'white_label' => 'boolean',
            'is_active' => 'boolean',
            'validity_days' => 'integer',
        ];
    }

    /**
     * Get the subscriptions for this plan.
     */
    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class, 'plan_id');
    }

    /**
     * Get price for billing cycle
     */
    public function getPriceForCycle(string $cycle): ?float
    {
        return match ($cycle) {
            'monthly' => $this->monthly_price,
            'quarterly' => $this->quarterly_price,
            'annual' => $this->annual_price,
            default => $this->monthly_price,
        };
    }

    /**
     * Get all enabled features
     */
    public function getEnabledFeatures(): array
    {
        $features = [];

        if ($this->sms_enabled) {
            $features[] = 'SMS';
        }
        if ($this->whatsapp_enabled) {
            $features[] = 'WhatsApp';
        }
        if ($this->email_enabled) {
            $features[] = 'Email';
        }
        if ($this->fcm_enabled) {
            $features[] = 'Push Notifications (FCM)';
        }
        if ($this->advanced_reports) {
            $features[] = 'Advanced Reports';
        }
        if ($this->api_access) {
            $features[] = 'API Access';
        }
        if ($this->custom_branding) {
            $features[] = 'Custom Branding';
        }
        if ($this->priority_support) {
            $features[] = 'Priority Support';
        }
        if ($this->white_label) {
            $features[] = 'White Label';
        }

        return $features;
    }
}
