<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Subscription extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'plan_id',
        'plan_name', // Keep for backward compatibility
        'plan_display_name', // Keep for backward compatibility
        'monthly_price', // Keep for backward compatibility
        'annual_price', // Keep for backward compatibility
        'billing_cycle',
        'status',
        'trial_start_date',
        'trial_end_date',
        'start_date',
        'end_date',
        'next_billing_date',
        'payment_gateway',
        'gateway_subscription_id',
        'gateway_customer_id',
        'max_users', // Keep for backward compatibility
        'max_units', // Keep for backward compatibility
        'max_buildings', // Keep for backward compatibility
        'sms_enabled', // Keep for backward compatibility
        'whatsapp_enabled', // Keep for backward compatibility
        'email_enabled', // Keep for backward compatibility
        'fcm_enabled', // Keep for backward compatibility
        'advanced_reports', // Keep for backward compatibility
        'api_access', // Keep for backward compatibility
        'storage_gb', // Keep for backward compatibility
        'auto_renew',
        'cancelled_at',
        'cancellation_reason',
    ];

    protected function casts(): array
    {
        return [
            'monthly_price' => 'decimal:2',
            'annual_price' => 'decimal:2',
            'trial_start_date' => 'date',
            'trial_end_date' => 'date',
            'start_date' => 'date',
            'end_date' => 'date',
            'next_billing_date' => 'date',
            'sms_enabled' => 'boolean',
            'whatsapp_enabled' => 'boolean',
            'email_enabled' => 'boolean',
            'fcm_enabled' => 'boolean',
            'advanced_reports' => 'boolean',
            'api_access' => 'boolean',
            'auto_renew' => 'boolean',
            'cancelled_at' => 'datetime',
        ];
    }

    /**
     * Get the society that owns the subscription.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the subscription plan.
     */
    public function plan(): BelongsTo
    {
        return $this->belongsTo(SubscriptionPlan::class, 'plan_id');
    }

    /**
     * Get plan features (from plan or fallback to subscription fields)
     */
    public function hasFeature(string $feature): bool
    {
        if ($this->plan) {
            return match ($feature) {
                'sms' => $this->plan->sms_enabled,
                'whatsapp' => $this->plan->whatsapp_enabled,
                'email' => $this->plan->email_enabled,
                'fcm' => $this->plan->fcm_enabled,
                'advanced_reports' => $this->plan->advanced_reports,
                'api_access' => $this->plan->api_access,
                default => false,
            };
        }

        // Fallback to subscription fields for backward compatibility
        return match ($feature) {
            'sms' => $this->sms_enabled,
            'whatsapp' => $this->whatsapp_enabled,
            'email' => $this->email_enabled,
            'fcm' => $this->fcm_enabled,
            'advanced_reports' => $this->advanced_reports,
            'api_access' => $this->api_access,
            default => false,
        };
    }

    /**
     * Check if subscription is active
     */
    public function isActive(): bool
    {
        return $this->status === 'active' &&
               ($this->end_date === null || $this->end_date >= now()->toDateString());
    }

    /**
     * Check if subscription is in trial
     */
    public function isTrial(): bool
    {
        return $this->status === 'trial' &&
               $this->trial_end_date &&
               $this->trial_end_date >= now()->toDateString();
    }

    /**
     * Check if subscription is pending payment
     */
    public function isPendingPayment(): bool
    {
        return $this->status === 'pending_payment';
    }

    /**
     * Check if subscription is expired
     */
    public function isExpired(): bool
    {
        if (! $this->end_date) {
            return false;
        }

        return $this->end_date < now()->toDateString();
    }

    /**
     * Check if subscription is expired but within grace period (5 days)
     */
    public function isExpiredInGracePeriod(): bool
    {
        if (! $this->isExpired()) {
            return false;
        }

        $gracePeriodEnd = $this->getGracePeriodEndDate();

        return $gracePeriodEnd && $gracePeriodEnd >= now()->toDateString();
    }

    /**
     * Check if subscription is expired and beyond grace period
     */
    public function isExpiredBeyondGracePeriod(): bool
    {
        if (! $this->isExpired()) {
            return false;
        }

        $gracePeriodEnd = $this->getGracePeriodEndDate();

        return ! $gracePeriodEnd || $gracePeriodEnd < now()->toDateString();
    }

    /**
     * Get grace period end date (5 days after expiry)
     */
    public function getGracePeriodEndDate(): ?\Carbon\Carbon
    {
        if (! $this->end_date) {
            return null;
        }

        return \Carbon\Carbon::parse($this->end_date)->addDays(5);
    }

    /**
     * Get remaining days in grace period
     */
    public function getGracePeriodDaysRemaining(): int
    {
        if (! $this->isExpiredInGracePeriod()) {
            return 0;
        }

        $gracePeriodEnd = $this->getGracePeriodEndDate();

        if (! $gracePeriodEnd) {
            return 0;
        }

        return max(0, now()->diffInDays($gracePeriodEnd, false));
    }
}
