<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class SocietySetting extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'payment_gateway',
        'razorpay_key_id',
        'razorpay_key_secret',
        'razorpay_webhook_secret',
        'razorpay_enabled',
        'stripe_public_key',
        'stripe_secret_key',
        'stripe_enabled',
        'payu_merchant_key',
        'payu_merchant_salt',
        'payu_enabled',
        'cashfree_app_id',
        'cashfree_secret_key',
        'cashfree_enabled',
        'email_enabled',
        'email_provider',
        'smtp_host',
        'smtp_port',
        'smtp_username',
        'smtp_password',
        'smtp_encryption',
        'from_email',
        'from_name',
        'sms_enabled',
        'sms_provider',
        'sms_api_key',
        'sms_sender_id',
        'sms_username',
        'sms_password',
        'whatsapp_enabled',
        'whatsapp_provider',
        'whatsapp_api_key',
        'whatsapp_sender_id',
        'whatsapp_business_id',
        'fcm_enabled',
        'fcm_server_key',
        'fcm_sender_id',
        'timezone',
        'currency',
        'late_fee_calculation_type', // [NEW]
        'late_fee_value',            // [NEW]
        'date_format',
        'time_format',
        'language',
        'notify_bill_generated',
        'notify_payment_received',
        'notify_visitor_arrival',
        'notify_maintenance_request',
        'notify_announcement',
        'enable_visitor_management',
        'enable_maintenance_tickets',
        'enable_complaints',
        'enable_events',
        'enable_documents',
        'enable_directory',
        'enable_notices',
        'enable_polls',
        'enable_marketplace',
        'enable_facility_booking',
        'require_otp_for_login',
        'require_otp_for_payment',
        'session_timeout_minutes',
        'enable_2fa',
        'google_analytics_id',
        'facebook_pixel_id',
        'custom_css',
        'custom_js',
    ];

    protected function casts(): array
    {
        return [
            'razorpay_enabled' => 'boolean',
            'stripe_enabled' => 'boolean',
            'payu_enabled' => 'boolean',
            'cashfree_enabled' => 'boolean',
            'email_enabled' => 'boolean',
            'sms_enabled' => 'boolean',
            'whatsapp_enabled' => 'boolean',
            'fcm_enabled' => 'boolean',
            'notify_bill_generated' => 'boolean',
            'notify_payment_received' => 'boolean',
            'notify_visitor_arrival' => 'boolean',
            'notify_maintenance_request' => 'boolean',
            'notify_announcement' => 'boolean',
            'enable_visitor_management' => 'boolean',
            'enable_maintenance_tickets' => 'boolean',
            'enable_complaints' => 'boolean',
            'enable_events' => 'boolean',
            'enable_documents' => 'boolean',
            'enable_directory' => 'boolean',
            'enable_notices' => 'boolean',
            'enable_polls' => 'boolean',
            'enable_marketplace' => 'boolean',
            'enable_facility_booking' => 'boolean',
            'require_otp_for_login' => 'boolean',
            'require_otp_for_payment' => 'boolean',
            'enable_2fa' => 'boolean',
            'smtp_port' => 'integer',
            'session_timeout_minutes' => 'integer',
        ];
    }

    /**
     * Get the society that owns the settings.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get payment gateway config
     */
    public function getPaymentGatewayConfig(): ?array
    {
        return match ($this->payment_gateway) {
            'razorpay' => [
                'key_id' => $this->razorpay_key_id,
                'key_secret' => $this->razorpay_key_secret,
                'enabled' => $this->razorpay_enabled,
            ],
            'stripe' => [
                'public_key' => $this->stripe_public_key,
                'secret_key' => $this->stripe_secret_key,
                'enabled' => $this->stripe_enabled,
            ],
            'payu' => [
                'merchant_key' => $this->payu_merchant_key,
                'merchant_salt' => $this->payu_merchant_salt,
                'enabled' => $this->payu_enabled,
            ],
            'cashfree' => [
                'app_id' => $this->cashfree_app_id,
                'secret_key' => $this->cashfree_secret_key,
                'enabled' => $this->cashfree_enabled,
            ],
            default => null,
        };
    }

    /**
     * Check if broadcast medium is enabled
     */
    public function isBroadcastEnabled(string $medium): bool
    {
        return match ($medium) {
            'email' => $this->email_enabled,
            'sms' => $this->sms_enabled,
            'whatsapp' => $this->whatsapp_enabled,
            'fcm' => $this->fcm_enabled,
            default => false,
        };
    }
}
