<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class Society extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'registration_no',
        'address',
        'city',
        'state',
        'pincode',
        'status',
        'logo',
        'gst_no',
        'pan_no',
        'rules',
        'code',
        'phone',
        'email',
    ];

    protected function casts(): array
    {
        return [
            'rules' => 'array',
        ];
    }

    // ========== SUPER ADMIN SCOPES ==========

    /**
     * Scope: Active societies only
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope: Pending platform fee societies
     */
    public function scopePendingFee(Builder $query): Builder
    {
        // TODO: Update logic based on subscription status instead of platform fee
        return $query->whereHas('subscription', function ($q) {
            $q->where('status', 'pending'); // Assuming pending status exists for subscription
        });
    }

    /**
     * Scope: Recent societies (last 30 days)
     */
    public function scopeRecent(Builder $query, int $days = 30): Builder
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Scope: Search by name/registration
     */
    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
                ->orWhere('registration_no', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Scope: By status filter
     */
    public function scopeStatusFilter(Builder $query, array $statuses): Builder
    {
        return $query->whereIn('status', $statuses);
    }

    /**
     * Scope: Revenue stats (Super Admin Dashboard)
     */
    public function scopeRevenueStats(Builder $query, string $period = 'month'): Builder
    {
        // TODO: Update logic based on subscription payments
        return $query; 
    }

    // ========== RELATIONSHIPS ==========

    /**
     * Get the admin for the society.
     */
    public function admin(): HasOne
    {
        return $this->hasOne(User::class)->whereHas('role', function ($q) {
            $q->where('name', 'society_admin');
        })->latest();
    }

    /**
     * Get the subscription for the society.
     */
    public function subscription(): HasOne
    {
        return $this->hasOne(Subscription::class, 'society_id');
    }

    /**
     * Get the settings for the society.
     */
    public function settings(): HasOne
    {
        return $this->hasOne(SocietySetting::class, 'society_id');
    }

    /**
     * Get the users for the society.
     */
    public function users(): HasMany
    {
        return $this->hasMany(User::class);
    }

    /**
     * Get the buildings for the society.
     */
    public function buildings(): HasMany
    {
        return $this->hasMany(Building::class);
    }

    /**
     * Get the units for the society.
     */
    public function units(): HasMany
    {
        return $this->hasMany(Unit::class);
    }

    /**
     * Get the bills for the society.
     */
    public function bills(): HasMany
    {
        return $this->hasMany(Bill::class);
    }

    /**
     * Get the bill cycles for the society.
     */
    public function billCycles(): HasMany
    {
        return $this->hasMany(BillCycle::class);
    }

    /**
     * Get the payments for the society.
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the expenses for the society.
     */
    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class);
    }

    /**
     * Get the maintenance tickets for the society.
     */
    public function maintenanceTickets(): HasMany
    {
        return $this->hasMany(MaintenanceTicket::class);
    }

    /**
     * Get the visitors for the society.
     */
    public function visitors(): HasMany
    {
        return $this->hasMany(Visitor::class);
    }

    /**
     * Get the bank accounts for the society.
     */
    public function bankAccounts(): HasMany
    {
        return $this->hasMany(BankAccount::class);
    }

    /**
     * Get the announcements for the society.
     */
    public function announcements(): HasMany
    {
        return $this->hasMany(Announcement::class);
    }

    /**
     * Get the vendors for the society.
     */
    public function vendors(): HasMany
    {
        return $this->hasMany(Vendor::class);
    }

    /**
     * Get the community groups for the society.
     */
    public function communityGroups(): HasMany
    {
        return $this->hasMany(CommunityGroup::class);
    }

    /**
     * Get the maintenance structures for the society.
     */
    public function maintenanceStructures(): HasMany
    {
        return $this->hasMany(MaintenanceStructure::class);
    }

    /**
     * Get the community posts for the society.
     */
    public function communityPosts(): HasMany
    {
        return $this->hasMany(CommunityPost::class);
    }

    /**
     * Get the facilities for the society.
     */
    public function facilities(): HasMany
    {
        return $this->hasMany(Facility::class);
    }

    /**
     * Get the facility bookings for the society.
     */
    public function facilityBookings(): HasMany
    {
        return $this->hasMany(FacilityBooking::class);
    }


    /**
     * Get the platform fee (latest payment) for the society.
     */
    public function platformFee(): HasOne
    {
        // TODO: Filter for specific platform fee payments if distinguished by type/notes
        return $this->hasOne(Payment::class)->latest();
    }

    /**
     * Get the platform user (society admin) for the society.
     */
    public function platformUser(): HasOne
    {
        return $this->admin();
    }

    // ========== SUPER ADMIN ACCESSORS ==========

    /**
     * Get society full address
     */
    public function getFullAddressAttribute(): string
    {
        return "{$this->address}, {$this->city}, {$this->state} - {$this->pincode}";
    }

    /**
     * Check if society is revenue generating
     */
    public function getIsRevenueActiveAttribute(): bool
    {
        return $this->platformFee && $this->platformFee->status === 'paid';
    }

    /**
     * Fee status badge color
     */
    public function getFeeStatusColorAttribute(): string
    {
        return match ($this->platformFee?->status) {
            'paid' => 'success',
            'pending' => 'warning',
            'overdue' => 'danger',
            default => 'gray',
        };
    }

    // ========== HELPER METHODS ==========

    /**
     * Check if society has a specific feature enabled via subscription
     */
    public function hasFeature(string $feature): bool
    {
        return $this->subscription?->hasFeature($feature) ?? false;
    }

    /**
     * Check if a feature limit has been reached
     */
    public function isFeatureLimitReached(string $feature): bool
    {
        if (! $this->subscription) {
            return true;
        }

        // Get limit from subscription (or plan via subscription if needed, 
        // but Subscription model has these fields)
        $limit = match ($feature) {
            'users' => $this->subscription->max_users,
            'units' => $this->subscription->max_units,
            'buildings' => $this->subscription->max_buildings,
            default => null,
        };

        if ($limit === null) {
            return false;
        }

        // If limit is 0, it might mean unlimited or not allowed. 
        // Assuming typical SaaS logic: 0 = not allowed, -1 = unlimited.
        // But based on migration defaults (usually 0 or specific number), let's assume > 0 is limit.
        // If 0, then limit is reached immediately.
        
        $currentCount = match ($feature) {
            'users' => $this->users()->count(),
            'units' => $this->units()->count(),
            'buildings' => $this->buildings()->count(),
            default => 0,
        };

        return $currentCount >= $limit;
    }

    /**
     * Get collection rate percentage
     */
    public function getCollectionRate(): float
    {
        $totalBilled = $this->bills()
            ->where('status', '!=', 'cancelled')
            ->sum('amount');

        if ($totalBilled <= 0) {
            return 0.0;
        }

        $totalCollected = $this->bills()
            ->where('status', 'paid')
            ->sum('amount');

        return round(($totalCollected / $totalBilled) * 100, 2);
    }
}
