<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class MaintenanceTicket extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'user_id',
        'ticket_no',
        'title',
        'description',
        'priority',
        'status',
        'vendor_id',
        'assigned_at',
        'resolved_at',
        'resolution_notes',
    ];

    protected function casts(): array
    {
        return [
            'assigned_at' => 'datetime',
            'resolved_at' => 'datetime',
        ];
    }

    /**
     * Get the society that owns the maintenance ticket.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the user (resident) who created the ticket.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the vendor (user) assigned to the ticket.
     */
    public function vendor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'vendor_id');
    }

    /**
     * Get the service vendor (Vendor model) assigned to the ticket.
     */
    public function serviceVendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class, 'vendor_id');
    }

    /**
     * Get the comments for the ticket.
     */
    public function comments(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Comment::class, 'ticket_id');
    }

    /**
     * Get the timeline entries for the ticket.
     */
    public function timelines(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(TicketTimeline::class, 'ticket_id');
    }

    /**
     * Record a timeline entry.
     */
    public function recordTimeline(string $action, string $description, ?int $userId = null)
    {
        $this->timelines()->create([
            'action' => $action,
            'description' => $description,
            'user_id' => $userId ?? auth()->id(),
        ]);
    }
}
