<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class FacilitySlot extends Model
{
    use HasFactory;

    protected $fillable = [
        'facility_id',
        'name',
        'start_time',
        'end_time',
        'price',
        'max_bookings_per_slot',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'start_time' => 'datetime:H:i:s',
            'end_time' => 'datetime:H:i:s',
            'price' => 'decimal:2',
            'max_bookings_per_slot' => 'integer',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the facility that owns the slot.
     */
    public function facility(): BelongsTo
    {
        return $this->belongsTo(Facility::class);
    }

    /**
     * Get the bookings for this slot.
     */
    public function bookings(): HasMany
    {
        return $this->hasMany(FacilityBooking::class, 'slot_id');
    }

    /**
     * Get formatted time range.
     */
    public function getTimeRangeAttribute(): string
    {
        return $this->start_time->format('h:i A') . ' - ' . $this->end_time->format('h:i A');
    }

    /**
     * Check if slot is available for a specific date.
     */
    public function isAvailableOn($date): bool
    {
        if (!$this->is_active) {
            return false;
        }

        $bookingCount = $this->bookings()
            ->where('booking_date', $date)
            ->whereNotIn('booking_status', ['cancelled', 'rejected'])
            ->count();

        return $bookingCount < $this->max_bookings_per_slot;
    }
}
