<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class FacilityBooking extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'facility_id',
        'slot_id',
        'user_id',
        'booked_by',
        'booking_date',
        'amount',
        'payment_status',
        'payment_method',
        'booking_status',
        'purpose',
        'guest_count',
        'admin_notes',
        'cancellation_reason',
        'checked_in_at',
        'checked_out_at',
        'payment_details',
    ];

    protected function casts(): array
    {
        return [
            'booking_date' => 'date',
            'amount' => 'decimal:2',
            'guest_count' => 'integer',
            'checked_in_at' => 'datetime',
            'checked_out_at' => 'datetime',
            'payment_details' => 'array',
        ];
    }

    /**
     * Get the society that owns the booking.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the facility for this booking.
     */
    public function facility(): BelongsTo
    {
        return $this->belongsTo(Facility::class);
    }

    /**
     * Get the slot for this booking.
     */
    public function slot(): BelongsTo
    {
        return $this->belongsTo(FacilitySlot::class, 'slot_id');
    }

    /**
     * Get the user (resident) who this booking is for.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the user who made the booking (could be admin or collector).
     */
    public function bookedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'booked_by');
    }

    /**
     * Scope: Filter by booking status.
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('booking_status', $status);
    }

    /**
     * Scope: Filter by payment status.
     */
    public function scopePaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    /**
     * Scope: Upcoming bookings.
     */
    public function scopeUpcoming($query)
    {
        return $query->where('booking_date', '>=', now()->toDateString())
            ->whereNotIn('booking_status', ['cancelled', 'rejected', 'completed']);
    }

    /**
     * Scope: Past bookings.
     */
    public function scopePast($query)
    {
        return $query->where('booking_date', '<', now()->toDateString())
            ->orWhere('booking_status', 'completed');
    }

    /**
     * Check if booking can be cancelled.
     */
    public function canBeCancelled(): bool
    {
        // Can't cancel if already cancelled, rejected, or completed
        if (in_array($this->booking_status, ['cancelled', 'rejected', 'completed'])) {
            return false;
        }

        // Can't cancel if the booking date has passed
        if ($this->booking_date < now()->toDateString()) {
            return false;
        }

        return true;
    }

    /**
     * Get status badge color.
     */
    public function getStatusColorAttribute(): string
    {
        return match ($this->booking_status) {
            'confirmed' => 'success',
            'pending' => 'warning',
            'cancelled', 'rejected' => 'danger',
            'completed' => 'info',
            default => 'secondary',
        };
    }

    /**
     * Get payment status badge color.
     */
    public function getPaymentStatusColorAttribute(): string
    {
        return match ($this->payment_status) {
            'paid' => 'success',
            'pending' => 'warning',
            'failed' => 'danger',
            'refunded' => 'info',
            default => 'secondary',
        };
    }

    public function transactions(): \Illuminate\Database\Eloquent\Relations\MorphMany
    {
        return $this->morphMany(Transaction::class, 'reference');
    }
}
