<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Facility extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'name',
        'description',
        'images',
        'capacity',
        'amenities',
        'rules',
        'cancellation_policy',
        'is_active',
        'requires_approval',
    ];

    protected function casts(): array
    {
        return [
            'images' => 'array',
            'is_active' => 'boolean',
            'requires_approval' => 'boolean',
            'capacity' => 'integer',
        ];
    }

    /**
     * Get the society that owns the facility.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the slots for the facility.
     */
    public function slots(): HasMany
    {
        return $this->hasMany(FacilitySlot::class);
    }

    /**
     * Get the bookings for the facility.
     */
    public function bookings(): HasMany
    {
        return $this->hasMany(FacilityBooking::class);
    }

    /**
     * Get active slots only.
     */
    public function activeSlots(): HasMany
    {
        return $this->slots()->where('is_active', true);
    }

    /**
     * Check if a specific date and slot is available for booking.
     */
    public function isSlotAvailable($slotId, $date): bool
    {
        $slot = $this->slots()->find($slotId);
        
        if (!$slot || !$slot->is_active) {
            return false;
        }

        // Count existing bookings for this date and slot
        $bookingCount = $this->bookings()
            ->where('slot_id', $slotId)
            ->where('booking_date', $date)
            ->whereNotIn('booking_status', ['cancelled', 'rejected'])
            ->count();

        return $bookingCount < $slot->max_bookings_per_slot;
    }
}
