<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CollectorWallet extends Model
{
    use HasFactory;

    protected $fillable = [
        'collector_id',
        'society_id',
        'balance',
        'total_collected',
        'total_handover',
    ];

    protected $casts = [
        'balance' => 'decimal:2',
        'total_collected' => 'decimal:2',
        'total_handover' => 'decimal:2',
    ];

    /**
     * Get the collector (user) that owns the wallet.
     */
    public function collector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'collector_id');
    }

    /**
     * Get the society this wallet belongs to.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get all transactions for this wallet.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(CollectorTransaction::class, 'wallet_id');
    }

    /**
     * Credit the wallet (add money - cash collection).
     */
    public function credit(float $amount, ?int $paymentId = null, ?string $notes = null): CollectorTransaction
    {
        $this->balance += $amount;
        $this->total_collected += $amount;
        $this->save();

        return $this->transactions()->create([
            'collector_id' => $this->collector_id,
            'society_id' => $this->society_id,
            'type' => 'credit',
            'amount' => $amount,
            'balance_after' => $this->balance,
            'reference_type' => $paymentId ? 'payment' : null,
            'reference_id' => $paymentId,
            'notes' => $notes ?? 'Cash collected',
        ]);
    }

    /**
     * Debit the wallet (remove money - handover).
     */
    public function debit(float $amount, int $receivedById, ?string $notes = null): CollectorTransaction
    {
        if ($amount > $this->balance) {
            throw new \Exception('Insufficient wallet balance.');
        }

        $this->balance -= $amount;
        $this->total_handover += $amount;
        $this->save();

        return $this->transactions()->create([
            'collector_id' => $this->collector_id,
            'society_id' => $this->society_id,
            'type' => 'debit',
            'amount' => $amount,
            'balance_after' => $this->balance,
            'reference_type' => 'handover',
            'received_by' => $receivedById,
            'notes' => $notes ?? 'Cash handed over',
        ]);
    }

    /**
     * Get or create wallet for a collector.
     */
    public static function getOrCreate(int $collectorId, int $societyId): self
    {
        return self::firstOrCreate(
            ['collector_id' => $collectorId, 'society_id' => $societyId],
            ['balance' => 0, 'total_collected' => 0, 'total_handover' => 0]
        );
    }
}
