<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CollectorTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'collector_id',
        'society_id',
        'wallet_id',
        'type',
        'amount',
        'balance_after',
        'reference_type',
        'reference_id',
        'received_by',
        'notes',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_after' => 'decimal:2',
    ];

    /**
     * Get the collector (user).
     */
    public function collector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'collector_id');
    }

    /**
     * Get the wallet.
     */
    public function wallet(): BelongsTo
    {
        return $this->belongsTo(CollectorWallet::class, 'wallet_id');
    }

    /**
     * Get the society.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the admin/accountant who received the handover.
     */
    public function receiver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Get the payment reference (if credit from payment).
     */
    public function payment(): BelongsTo
    {
        return $this->belongsTo(Payment::class, 'reference_id');
    }

    /**
     * Scope for credits only.
     */
    public function scopeCredits($query)
    {
        return $query->where('type', 'credit');
    }

    /**
     * Scope for debits only.
     */
    public function scopeDebits($query)
    {
        return $query->where('type', 'debit');
    }
}
