<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class Bill extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'bill_cycle_id',
        'user_id',
        'bill_no',
        'amount',
        'maintenance',
        'water',
        'electricity',
        'parking',
        'other_charges',
        'gst',
        'discount',
        'status',
        'due_date',
        'paid_at',
        'generated_by',
        'revised_by',
        'revised_at',
        'revision_note',
        'receipt_path',
        'late_fee',                 // [NEW]
        'last_interest_applied_at', // [NEW]
    ];

    protected function casts(): array
    {
        return [
            'amount' => 'decimal:2',
            'maintenance' => 'decimal:2',
            'water' => 'decimal:2',
            'electricity' => 'decimal:2',
            'parking' => 'decimal:2',
            'other_charges' => 'decimal:2',
            'gst' => 'decimal:2',
            'discount' => 'decimal:2',
            'late_fee' => 'decimal:2', // [NEW]
            'due_date' => 'date',
            'paid_at' => 'date',
            'revised_at' => 'datetime',
            'last_interest_applied_at' => 'date', // [NEW]
        ];
    }

    /**
     * Get the total amount (principal + late fee).
     */
    public function getTotalAmountAttribute(): float
    {
        return $this->amount + ($this->late_fee ?? 0);
    }

    /**
     * Get the society that owns the bill.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the bill cycle that owns the bill.
     */
    public function billCycle(): BelongsTo
    {
        return $this->belongsTo(BillCycle::class);
    }

    /**
     * Get the user (resident) that owns the bill.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the user who generated the bill.
     */
    public function generatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'generated_by');
    }

    /**
     * Get the user who revised the bill.
     */
    public function revisedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'revised_by');
    }

    /**
     * Get the payment for the bill.
     */
    public function payment(): HasOne
    {
        return $this->hasOne(Payment::class);
    }
}
