<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankAccount extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'account_name',
        'account_number',
        'ifsc_code',
        'bank_name',
        'branch_name',
        'balance',
        'account_type',
        'updated_by',
        'last_reconciled_at',
        'is_primary',
        'description',
    ];

    protected $casts = [
        'balance' => 'decimal:2',
        'last_reconciled_at' => 'datetime',
        'is_primary' => 'boolean',
    ];

    /**
     * Get the society that owns the bank account.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the user who last updated the bank account.
     */
    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get the payments associated with the bank account.
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the transactions for the bank account.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Mark this account as primary and unset others.
     */
    public function markAsPrimary()
    {
        \DB::transaction(function () {
            self::where('society_id', $this->society_id)
                ->where('id', '!=', $this->id)
                ->update(['is_primary' => false]);
            
            $this->update(['is_primary' => true]);
        });
    }
}
