<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Announcement extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'society_id',
        'title',
        'content',
        'type',
        'priority',
        'target_roles',
        'target_buildings',
        // 'target_units', // Keeping this for backward compatibility or direct unit targeting if needed, but UI primarily uses buildings now
        'target_units',
        'start_date',
        'end_date',
        'require_acknowledgement',
        'attachments',
        'views_count',
        'acknowledged_count',
        'is_published',
        'created_by',
    ];

    protected function casts(): array
    {
        return [
            'target_roles' => 'array',
            'target_buildings' => 'array',
            'target_units' => 'array',
            'start_date' => 'datetime',
            'end_date' => 'datetime',
            'attachments' => 'array',
            'require_acknowledgement' => 'boolean',
            'is_published' => 'boolean',
        ];
    }

    /**
     * Get the society that owns the announcement.
     */
    public function society(): BelongsTo
    {
        return $this->belongsTo(Society::class);
    }

    /**
     * Get the user who created the announcement.
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the views for the announcement.
     */
    public function views(): HasMany
    {
        return $this->hasMany(AnnouncementView::class);
    }

    /**
     * Get the acknowledgements for the announcement.
     */
    public function acknowledgements(): HasMany
    {
        return $this->hasMany(AnnouncementAcknowledgement::class);
    }
}
