<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class BillResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'bill_no' => $this->bill_no,
            'amount' => number_format($this->amount, 2, '.', ''),
            'status' => in_array($this->status, ['due', 'approved']) ? 'pending' : $this->status,
            'due_date' => $this->due_date instanceof \DateTime ? $this->due_date->format('Y-m-d') : $this->due_date,
            'created_at' => $this->created_at->toIso8601String(),
            
            // Charge Breakdown
            'charges' => [
                'maintenance' => number_format($this->maintenance ?? 0, 2, '.', ''),
                'water' => number_format($this->water ?? 0, 2, '.', ''),
                'electricity' => number_format($this->electricity ?? 0, 2, '.', ''),
                'parking' => number_format($this->parking ?? 0, 2, '.', ''),
                'other_charges' => number_format($this->other_charges ?? 0, 2, '.', ''),
                'gst' => number_format($this->gst ?? 0, 2, '.', ''),
                'discount' => number_format($this->discount ?? 0, 2, '.', ''),
            ],
            
            // User/Resident Information
            'user' => $this->when($this->relationLoaded('user'), function() {
                return [
                    'id' => $this->user->id,
                    'name' => $this->user->name,
                    'email' => $this->user->email,
                    'phone' => $this->user->phone,
                    'unit' => $this->user->unit ? [
                        'id' => $this->user->unit->id,
                        'unit_no' => $this->user->unit->unit_no,
                        'building' => $this->user->unit->building ? [
                            'id' => $this->user->unit->building->id,
                            'name' => $this->user->unit->building->name,
                        ] : null,
                    ] : null,
                ];
            }),
            
            // Bill Cycle Information
            'bill_cycle' => $this->when($this->relationLoaded('billCycle') && $this->billCycle, function() {
                return [
                    'id' => $this->billCycle->id,
                    'name' => $this->billCycle->name,
                    'start_date' => $this->billCycle->start_date,
                    'end_date' => $this->billCycle->end_date,
                ];
            }),
            
            // Society Information (for invoice header)
            'society' => $this->when($this->relationLoaded('society'), function() {
                return [
                    'id' => $this->society->id,
                    'name' => $this->society->name,
                    'code' => $this->society->code,
                    'address' => $this->society->address,
                    'city' => $this->society->city,
                    'state' => $this->society->state,
                    'pincode' => $this->society->pincode,
                    'phone' => $this->society->phone,
                    'email' => $this->society->email,
                    'logo' => $this->society->logo ? asset('storage/' . $this->society->logo) : null,
                    'gst_no' => $this->society->gst_no,
                    'registration_no' => $this->society->registration_no,
                ];
            }),
            
            // Payment Information (if paid)
            'payment' => $this->when($this->relationLoaded('payment') && $this->payment, function() {
                return [
                    'id' => $this->payment->id,
                    'payment_no' => $this->payment->payment_no,
                    'amount' => number_format($this->payment->amount, 2, '.', ''),
                    'payment_method' => $this->payment->payment_method,
                    'transaction_id' => $this->payment->transaction_id,
                    'payment_date' => $this->payment->payment_date,
                    'status' => $this->payment->status,
                ];
            }),
        ];
    }
}
