<?php

namespace App\Http\Middleware;

use Illuminate\Foundation\Inspiring;
use Illuminate\Http\Request;
use Inertia\Middleware;

class HandleInertiaRequests extends Middleware
{
    /**
     * The root template that's loaded on the first page visit.
     *
     * @see https://inertiajs.com/server-side-setup#root-template
     *
     * @var string
     */
    protected $rootView = 'app';

    /**
     * Determines the current asset version.
     *
     * @see https://inertiajs.com/asset-versioning
     */
    public function version(Request $request): ?string
    {
        return parent::version($request);
    }

    /**
     * Define the props that are shared by default.
     *
     * @see https://inertiajs.com/shared-data
     *
     * @return array<string, mixed>
     */
    public function share(Request $request): array
    {
        [$message, $author] = str(Inspiring::quotes()->random())->explode('-');

        $user = $request->user();
        $superAdmin = $request->user('super_admin');
        $authenticatedUser = $user ?? $superAdmin;

        // Get user permissions if authenticated (skip if app not installed)
        $permissions = [];
        $menuItems = [];
        $installedFile = storage_path('app/private/installed.json');
        $isInstalled = file_exists($installedFile);
        
        if ($isInstalled) {
            try {
                // Only load permissions for regular users (User model), not SuperAdmin
                if ($user && $user instanceof \App\Models\User) {
                    // Load role relationship if not already loaded
                    if (! $user->relationLoaded('role')) {
                        $user->load('role');
                    }

                    if ($user->role) {
                        $permissions = $user->role->permissions()->pluck('name')->toArray();
                        $societyId = $user->society_id ?? null;
                        if ($societyId) {
                            $menuItems = \App\Helpers\MenuHelper::getMenuItemsForUser($user);
                        }
                    }
                }
            } catch (\Exception $e) {
                // Database tables don't exist yet (during installation)
                // Just return empty arrays
            }
        }

        // Get subscription warning if set by CheckSocietyAccess middleware
        $subscriptionWarning = $request->attributes->get('subscription_warning');

        // Get real notifications for Super Admin (skip if app not installed)
        $notifications = [];
        
        if ($isInstalled) {
            try {
                if ($authenticatedUser && isset($authenticatedUser['role']) && $authenticatedUser['role']['name'] === 'super_admin') {
                    // Subscription expiring soon (within 7 days)
                    $expiringSocieties = \App\Models\Society::whereHas('subscription', function ($query) {
                        $query->where('end_date', '<=', now()->addDays(7))
                              ->where('end_date', '>=', now());
                    })->count();
                    
                    if ($expiringSocieties > 0) {
                        $notifications[] = [
                            'id' => 'expiring-' . now()->timestamp,
                            'title' => 'Subscriptions Expiring Soon',
                            'message' => "{$expiringSocieties} " . ($expiringSocieties === 1 ? 'society subscription' : 'societies subscriptions') . " expiring within 7 days",
                            'time' => 'Today',
                            'unread' => true,
                            'type' => 'warning',
                        ];
                    }

                    // Pending platform fees
                    $pendingFees = \App\Models\Society::pendingFee()->count();
                    if ($pendingFees > 0) {
                        $notifications[] = [
                            'id' => 'pending-fees-' . now()->timestamp,
                            'title' => 'Pending Platform Fees',
                            'message' => "{$pendingFees} " . ($pendingFees === 1 ? 'society has' : 'societies have') . " pending platform fees",
                            'time' => 'Today',
                            'unread' => true,
                            'type' => 'warning',
                        ];
                    }

                    // Recent society registrations (last 24 hours)
                    $recentSocietiesCount = \App\Models\Society::where('created_at', '>=', now()->subDay())->count();
                    if ($recentSocietiesCount > 0) {
                        $notifications[] = [
                            'id' => 'new-societies-' . now()->timestamp,
                            'title' => 'New Society Registrations',
                            'message' => "{$recentSocietiesCount} new " . ($recentSocietiesCount === 1 ? 'society' : 'societies') . " registered in the last 24 hours",
                            'time' => 'Today',
                            'unread' => true,
                            'type' => 'success',
                        ];
                    }

                    // Recent users (last 24 hours)
                    $recentUsersCount = \App\Models\User::where('created_at', '>=', now()->subDay())->count();
                    if ($recentUsersCount > 0) {
                        $notifications[] = [
                            'id' => 'new-users-' . now()->timestamp,
                            'title' => 'New User Signups',
                            'message' => "{$recentUsersCount} new " . ($recentUsersCount === 1 ? 'user has' : 'users have') . " joined in the last 24 hours",
                            'time' => 'Today',
                            'unread' => false,
                            'type' => 'success',
                        ];
                    }
                }

                // Fetch Database Notifications for Society Admins and Users
                if ($user) {
                    foreach ($user->unreadNotifications()->latest()->take(10)->get() as $n) {
                        $notifications[] = [
                            'id' => $n->id,
                            'title' => $n->data['title'] ?? 'Notification',
                            'message' => $n->data['message'] ?? '',
                            'time' => $n->created_at->diffForHumans(),
                            'unread' => true,
                            'type' => $n->data['type'] ?? 'info',
                        ];
                    }
                }
            } catch (\Exception $e) {
                // Database tables don't exist yet (during installation)
                // Just return empty notifications array
            }
        }

        return [
            ...parent::share($request),
            'name' => config('app.name'),
            'quote' => ['message' => trim($message), 'author' => trim($author)],
            'auth' => [
                'user' => $authenticatedUser,
                'permissions' => $permissions,
                'menuItems' => $menuItems,
            ],
            'sidebarOpen' => ! $request->hasCookie('sidebar_state') || $request->cookie('sidebar_state') === 'true',
            'subscriptionWarning' => $subscriptionWarning,
            'subscriptionPaymentRequired' => $request->attributes->get('subscription_payment_required'),
            'notifications' => $notifications,
            'flash' => [
                'success' => $request->session()->get('success'),
                'error' => $request->session()->get('error'),
                'warning' => $request->session()->get('warning'),
                'info' => $request->session()->get('info'),
            ],
            'settings' => $this->getPublicSettings(),
        ];
    }

    /**
     * Get public settings, or return empty array if app is not installed
     */
    protected function getPublicSettings(): array
    {
        // Check if app is installed
        $installedFile = storage_path('app/private/installed.json');
        if (!file_exists($installedFile)) {
            return [];
        }

        // Try to get settings, return empty array if table doesn't exist
        try {
            return \App\Models\SystemSetting::where('is_public', true)
                ->get()
                ->mapWithKeys(fn ($item) => [$item->key => $item->value])
                ->toArray();
        } catch (\Exception $e) {
            // Table doesn't exist yet (during installation)
            return [];
        }
    }
}
