<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckSocietyAccess
{
    /**
     * Grace period in days after subscription expiry
     */
    private const GRACE_PERIOD_DAYS = 5;

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = Auth::user();

        // Skip check for super admins
        if (! $user || $user->role?->name === 'super_admin') {
            return $next($request);
        }

        // Skip check if user has no society (shouldn't happen, but safety check)
        if (! $user->society_id) {
            return $next($request);
        }

        $society = $user->society;

        // Check society status first (inactive or suspended)
        if (in_array($society->status, ['inactive', 'suspended'])) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            $message = match ($society->status) {
                'inactive' => 'Your society account is inactive. Please contact support.',
                'suspended' => 'Your society account has been suspended. Please contact support.',
                default => 'Access denied. Please contact support.',
            };

            return redirect()->route('login')
                ->with('error', $message);
        }

        // Check subscription status - reload subscription to get latest data
        $subscription = $society->subscription()->first();

        if (! $subscription) {
            // No subscription found - allow access but show warning
            return $next($request);
        }

        // If subscription is active and not expired, allow access
        if ($subscription->isActive() && !$subscription->isExpired()) {
            return $next($request);
        }

        // Check available payment gateways
        $paymentGateways = \App\Services\PaymentGateway\PaymentGatewayFactory::getGatewayConfigs();
        $hasPaymentGateway = !empty($paymentGateways);

        // Check if subscription is pending payment
        if ($subscription->status === 'pending_payment') {
            // Set attribute for frontend to show payment required UI
            $request->attributes->set('subscription_payment_required', [
                'status' => 'pending_payment',
                'message' => 'Please complete your subscription payment to access the platform. Contact your super admin for renewal.',
                'has_payment_gateway' => $hasPaymentGateway,
            ]);
        }

        // Check if subscription is in trial (trials not allowed)
        if ($subscription->status === 'trial') {
            // Set attribute for frontend to show payment required UI
            $request->attributes->set('subscription_payment_required', [
                'status' => 'trial',
                'message' => 'Your trial has ended. Please complete payment to continue using the platform. Contact your super admin for renewal.',
                'has_payment_gateway' => $hasPaymentGateway,
            ]);
        }

        // Check if subscription is expired (beyond grace period or within grace period)
        if ($subscription->isExpiredBeyondGracePeriod()) {
            // Set attribute for frontend to show payment required UI
            $request->attributes->set('subscription_payment_required', [
                'status' => 'expired',
                'message' => 'Your subscription has expired. Please renew your subscription to continue using the platform. Contact your super admin for renewal.',
                'has_payment_gateway' => $hasPaymentGateway,
            ]);
        } elseif ($subscription->isExpiredInGracePeriod()) {
            // Within grace period - set both warning and payment required
            $request->attributes->set('subscription_payment_required', [
                'status' => 'expired_grace',
                'message' => 'Your subscription has expired. You have ' . $subscription->getGracePeriodDaysRemaining() . ' days left to renew before losing access. Contact your super admin for renewal.',
                'has_payment_gateway' => $hasPaymentGateway,
            ]);
        }

        return $next($request);
    }
}
