<?php

namespace App\Http\Controllers;

use App\Models\FacilityBooking;
use App\Models\SocietySetting;
use App\Models\SubscriptionPayment;
use App\Models\SystemSetting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    /**
     * Handle Razorpay webhook
     * 
     * Security:
     * - This endpoint is public (as required by Razorpay), but MUST verify the
     *   `X-Razorpay-Signature` header using the configured webhook secret.
     */
    public function handleRazorpay(Request $request)
    {
        $webhookBody = $request->getContent();
        $signature = (string) $request->header('X-Razorpay-Signature', '');
        if ($signature === '') {
            Log::warning('Razorpay webhook rejected: missing signature header');
            return response()->json(['error' => 'Missing signature'], 401);
        }

        $event = json_decode($webhookBody, true);
        if (!is_array($event)) {
            Log::warning('Razorpay webhook rejected: invalid JSON');
            return response()->json(['error' => 'Invalid JSON'], 400);
        }

        // DB-driven webhook secret resolution (society-specific vs SaaS)
        $candidateSecrets = $this->getRazorpayWebhookSecrets($event);
        if (empty($candidateSecrets)) {
            Log::error('Razorpay webhook secret not configured in database', [
                'event' => $event['event'] ?? 'unknown',
            ]);
            return response()->json(['error' => 'Webhook secret not configured'], 500);
        }

        $isVerified = false;
        foreach ($candidateSecrets as $secret) {
            $expectedSignature = hash_hmac('sha256', $webhookBody, $secret);
            if (hash_equals($expectedSignature, $signature)) {
                $isVerified = true;
                break;
            }
        }

        if (!$isVerified) {
            Log::warning('Razorpay webhook rejected: invalid signature', [
                'event' => $event['event'] ?? 'unknown',
            ]);
            return response()->json(['error' => 'Invalid signature'], 401);
        }

        /**
         * Handle Razorpay webhook events
         * Processes payment events from Razorpay payment gateway
         */
        try {
            switch ($event['event'] ?? '') {
                case 'payment.captured':
                    $this->handlePaymentCaptured($event['payload']['payment']['entity']);
                    break;

                case 'payment.failed':
                    $this->handlePaymentFailed($event['payload']['payment']['entity']);
                    break;

                case 'payment.authorized':
                    $this->handlePaymentAuthorized($event['payload']['payment']['entity']);
                    break;

                case 'order.paid':
                    $this->handleOrderPaid($event['payload']['order']['entity']);
                    break;

                case 'payment_link.paid':
                    $this->handlePaymentLinkPaid($event['payload']['payment_link']['entity']);
                    break;

                default:
                    // Log unhandled events for monitoring (may indicate new event types)
                    Log::warning('Unhandled Razorpay webhook event', [
                        'event' => $event['event'] ?? 'unknown'
                    ]);
            }

            return response()->json(['status' => 'ok']);

        } catch (\Exception $e) {
            // Log webhook processing errors for debugging payment gateway issues
            Log::error('Razorpay webhook processing error', [
                'error' => $e->getMessage(),
                'event' => $event['event'] ?? 'unknown',
            ]);

            return response()->json(['error' => 'Processing failed'], 500);
        }
    }

    /**
     * Resolve Razorpay webhook secrets from DB.
     *
     * - Society payments: `society_settings.razorpay_webhook_secret`
     * - SaaS subscription payments: `system_settings[key=razorpay_webhook_secret]`
     *
     * We intentionally do NOT read any env vars here.
     *
     * @return string[] non-empty secrets
     */
    private function getRazorpayWebhookSecrets(array $event): array
    {
        $secrets = [];

        $notes = $this->extractNotes($event);
        $type = strtolower((string) ($notes['type'] ?? ''));
        $societyId = $notes['society_id'] ?? null;

        // If subscription payment, always verify with SaaS/system webhook secret
        if ($type === 'subscription_activation') {
            $systemSecret = (string) SystemSetting::getValue('razorpay_webhook_secret', '');
            if ($systemSecret !== '') {
                $secrets[] = $systemSecret;
            }
            return array_values(array_unique($secrets));
        }

        // If order id maps to a subscription payment, treat as SaaS
        $orderId = data_get($event, 'payload.order.entity.id')
            ?? data_get($event, 'payload.payment.entity.order_id')
            ?? data_get($event, 'payload.payment_link.entity.order_id');

        if (is_string($orderId) && $orderId !== '') {
            $subPayment = SubscriptionPayment::where('razorpay_order_id', $orderId)->first();
            if ($subPayment) {
                $systemSecret = (string) SystemSetting::getValue('razorpay_webhook_secret', '');
                if ($systemSecret !== '') {
                    $secrets[] = $systemSecret;
                }
                return array_values(array_unique($secrets));
            }
        }

        // Resolve society_id (from notes, user, booking, etc.)
        $resolvedSocietyId = null;

        if (is_numeric($societyId)) {
            $resolvedSocietyId = (int) $societyId;
        } elseif (is_string($societyId) && ctype_digit($societyId)) {
            $resolvedSocietyId = (int) $societyId;
        }

        if (!$resolvedSocietyId) {
            $userId = $notes['user_id'] ?? null;
            if (is_numeric($userId) || (is_string($userId) && ctype_digit($userId))) {
                $user = User::find((int) $userId);
                if ($user && $user->society_id) {
                    $resolvedSocietyId = (int) $user->society_id;
                }
            }
        }

        if (!$resolvedSocietyId) {
            // Facility booking webhook: try lookup by order/payment id
            $paymentId = data_get($event, 'payload.payment.entity.id');
            if (is_string($paymentId) && $paymentId !== '') {
                $booking = FacilityBooking::where('razorpay_payment_id', $paymentId)->first();
                if ($booking) {
                    $resolvedSocietyId = (int) $booking->society_id;
                }
            }
        }

        if (!$resolvedSocietyId && is_string($orderId) && $orderId !== '') {
            $booking = FacilityBooking::where('razorpay_order_id', $orderId)->first();
            if ($booking) {
                $resolvedSocietyId = (int) $booking->society_id;
            }
        }

        if ($resolvedSocietyId) {
            $societySecret = (string) SocietySetting::where('society_id', $resolvedSocietyId)
                ->value('razorpay_webhook_secret');
            if ($societySecret !== '') {
                $secrets[] = $societySecret;
            }
        }

        // Optional fallback: allow validating with SaaS secret if configured
        // (useful when multiple webhook accounts are not separated yet)
        $systemSecret = (string) SystemSetting::getValue('razorpay_webhook_secret', '');
        if ($systemSecret !== '') {
            $secrets[] = $systemSecret;
        }

        return array_values(array_unique(array_filter($secrets, fn ($s) => is_string($s) && $s !== '')));
    }

    /**
     * Extract Razorpay "notes" from different webhook entity shapes.
     */
    private function extractNotes(array $event): array
    {
        $paths = [
            'payload.payment.entity.notes',
            'payload.order.entity.notes',
            'payload.payment_link.entity.notes',
        ];

        foreach ($paths as $path) {
            $notes = data_get($event, $path);
            if (is_array($notes)) {
                return $notes;
            }
        }

        return [];
    }

    /**
     * Handle payment captured event
     */
    private function handlePaymentCaptured($payment)
    {
        $booking = FacilityBooking::where('razorpay_payment_id', $payment['id'])->first();

        if ($booking) {
            $booking->update([
                'payment_status' => 'paid',
                'booking_status' => 'confirmed'
            ]);

            // Check if transaction already exists to avoid duplicate entries
            $existingTx = \App\Models\Transaction::where('reference_type', \App\Models\FacilityBooking::class)
                ->where('reference_id', $booking->id)
                ->exists();

            if (!$existingTx) {
                // Determine primary bank account
                $primaryAccount = \App\Models\BankAccount::where('society_id', $booking->society_id)
                    ->where('is_primary', true)
                    ->first();

                // Fallback
                if (!$primaryAccount) {
                    $primaryAccount = \App\Models\BankAccount::where('society_id', $booking->society_id)->first();
                }

                if ($primaryAccount) {
                    app(\App\Services\TransactionService::class)->recordIncome(
                        $payment['amount'] / 100, // Razorpay amount is in paise
                        $primaryAccount->id,
                        'facility_booking',
                        $booking,
                        'online',
                        'Razorpay Webhook: ' . $payment['id'],
                        $booking->user_id // Payer
                    );
                }
            }
        } else {
            // Log warning when payment received but booking not found (may indicate data inconsistency)
            Log::warning('Booking not found for captured payment', [
                'payment_id' => $payment['id']
            ]);
        }
    }

    /**
     * Handle payment failed event
     */
    private function handlePaymentFailed($payment)
    {
        $booking = FacilityBooking::where('razorpay_payment_id', $payment['id'])->first();

        if ($booking) {
             $booking->update([
                'payment_status' => 'failed',
                'booking_status' => 'rejected' // Or pending logic if you retry? Usually failed.
            ]);
        }

        // Log payment failures for monitoring
        Log::warning('Payment failed via webhook', [
            'payment_id' => $payment['id'],
            'order_id' => $payment['order_id'] ?? null,
            'error_code' => $payment['error_code'] ?? null,
            'error_description' => $payment['error_description'] ?? null
        ]);
    }

    /**
     * Handle payment authorized event
     */
    private function handlePaymentAuthorized($payment)
    {
        $booking = FacilityBooking::where('razorpay_payment_id', $payment['id'])->first();

        // Payment authorized - booking will be confirmed when payment is captured
        // No logging needed for authorized state
    }

    /**
     * Handle order paid event
     */
    private function handleOrderPaid($order)
    {
        $booking = FacilityBooking::where('razorpay_order_id', $order['id'])->first();

        // Order paid event - booking confirmation handled by payment.captured event
        // No logging needed here

        // [NEW] Handle Resident Bill Payments
        // Check order notes for bill info
        if (isset($order['notes']) && isset($order['notes']['type']) && $order['notes']['type'] === 'bill_payment') {
            $billId = $order['notes']['bill_id'];
            $userId = $order['notes']['user_id'];
            
            // Order paid event - bill payment handled by payment.captured event
            // No logging needed here
            $bill = \App\Models\Bill::find($billId);
            $user = \App\Models\User::find($userId);

            if ($bill && $user) {
                // Check if payment already exists
                $existingPayment = \App\Models\Payment::where('transaction_id', $order['id']) // Or use payment ID if available in order? No, order has ID.
                    ->orWhereJsonContains('notes', $order['id']) // Weak check
                    ->first();
                
                // Better: Check if bill is already paid to avoid double counting
                // if ($bill->status === 'paid') return; 

                // We need the PAYMENT ID from Razorpay to record it properly. 
                // order.paid event usually has 'payload.payment.entity' but here we only have $order (entity).
                // Wait, handleWebhook receives the whole payload? No, usually dispatcher sends entity.
                // If this is order.paid, the entity is ORDER. It might not contain the payment ID directly in root.
                // However, usually we should use 'payment.captured' for full payment details.
                // But let's proceed with finding/creating payment if missing.
                
                // Actually, let's create a payment record if it doesn't exist.
                // We'll use order_id as transaction_id temporarily if payment id is missing, or fetch it.
                // But safely, let's just update the Bill status if needed.
                
                $totalPaid = \App\Models\Payment::where('bill_id', $bill->id)
                    ->where('status', 'completed')
                    ->sum('amount');
                
                if ($totalPaid >= $bill->amount) {
                    $bill->update(['status' => 'paid', 'paid_at' => now()]);
                } else {
                     // If we are strictly sure this order IS paid (amount matches), we can force update.
                     // But we should record the payment.
                     // Let's defer to 'payment.captured' for recording the payment row if possible.
                     // But if we want redundancy:
                     
                     // 1. Create Payment if not exists
                     // We need a unique identifier. Order ID is unique.
                     $txnId = $order['id']; // Using Order ID as proxy if Payment ID unknown
                     
                     $existing = \App\Models\Payment::where('notes', 'like', "%$txnId%")->exists();
                     
                     if (!$existing) {
                         $amt = $order['amount_paid'] / 100;
                         if ($amt > 0) {
                            $payment = \App\Models\Payment::create([
                                'society_id' => $user->society_id,
                                'user_id' => $user->id,
                                'bill_id' => $bill->id,
                                'amount' => $amt,
                                'payment_method' => 'online',
                                'status' => 'completed', // Confirmed by webhook
                                'payment_no' => 'PAY-ONL-WH-' . time(),
                                'transaction_id' => $txnId, // Order ID
                                'notes' => 'Online Bill Payment via Razorpay Webhook (Order: ' . $txnId . ')',
                            ]);
                            
                            // Update Bill
                            $totalPaidNow = $totalPaid + $amt;
                            if ($totalPaidNow >= $bill->amount) {
                                $bill->update(['status' => 'paid', 'paid_at' => now()]);
                            } else {
                                $bill->update(['status' => 'partial']);
                            }

                            // Record Transaction
                            $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)->where('is_primary', true)->first();
                            if (!$primaryAccount) $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)->first();
                            
                            if ($primaryAccount) {
                                $payment->update(['bank_account_id' => $primaryAccount->id]);
                                app(\App\Services\TransactionService::class)->recordIncome(
                                    $payment->amount,
                                    $primaryAccount->id,
                                    'bill_payment',
                                    $payment,
                                    'online',
                                    'Online Bill Payment (Webhook): ' . $txnId,
                                    $user->id
                                );
                            }
                         }
                     }
                }
            }
        }
    }

    /**
     * Handle Payment Link Paid webhook event
     * Processes payment completion for UPI payment requests and bill payments
     */
    private function handlePaymentLinkPaid($paymentLink)
    {
        $paymentRequestId = $paymentLink['id'];

        // Find pending payment record linked to this payment request
        $payment = \App\Models\Payment::where('payment_request_id', $paymentRequestId)
            ->where('status', '!=', 'completed')
            ->first();

        if ($payment) {

            // Extract actual payment ID from Razorpay if available
            $transactionId = null;
            if (isset($paymentLink['payments']) && is_array($paymentLink['payments']) && count($paymentLink['payments']) > 0) {
                $transactionId = $paymentLink['payments'][0]['payment_id'] ?? null;
                // Extract transaction ID from Razorpay payment
            }

            // Update payment status to completed
            $updateData = [
                'status' => 'completed',
                'payment_date' => now(),
            ];
            
            if ($transactionId) {
                $updateData['transaction_id'] = $transactionId;
            } else {
                $updateData['transaction_id'] = $paymentRequestId;
            }
            
            $payment->update($updateData);
            
            // Update bill status based on total payments
            $bill = $payment->bill;
            if ($bill) {
                $totalPaid = \App\Models\Payment::where('bill_id', $bill->id)
                    ->where('status', 'completed')
                    ->sum('amount');
                
                // Update bill status based on payment amount
                if ($totalPaid >= $bill->amount) {
                    $bill->update(['status' => 'paid', 'paid_at' => now()]);
                } else {
                    $bill->update(['status' => 'partial']);
                }
            }

            // Record Transaction in Ledger
            $primaryAccount = \App\Models\BankAccount::where('society_id', $payment->society_id)
                ->where('is_primary', true)
                ->first();

            if (!$primaryAccount) {
                $primaryAccount = \App\Models\BankAccount::where('society_id', $payment->society_id)->first();
            }

            if ($primaryAccount) {
                $payment->update(['bank_account_id' => $primaryAccount->id]);

                // Record transaction in ledger
                app(\App\Services\TransactionService::class)->recordIncome(
                    $payment->amount,
                    $primaryAccount->id,
                    'bill_payment',
                    $payment,
                    'online_request',
                    'Online Payment Link Paid: ' . $paymentRequestId,
                    $payment->user_id
                );
            } else {
                // Log warning when no bank account found (transaction cannot be recorded)
                Log::warning('Payment Link Paid: No bank account found for transaction recording', [
                    'society_id' => $payment->society_id,
                ]);
            }

        } else {
            // Log warning when payment link paid but no matching payment record found
            Log::warning('Payment Link Paid: No matching payment record found', [
                'request_id' => $paymentRequestId,
            ]);
        }
    }
}
