<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\StoreUserRequest;
use App\Http\Requests\SuperAdmin\UpdateUserRequest;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Inertia\Response;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): Response
    {
        // 1. Fetch Societies with their Users (filtered)
        $societiesQuery = Society::query()
            ->with(['users' => function ($query) use ($request) {
                $query->with('role');
                
                // Apply filters to the Users query
                if ($request->has('search')) {
                    $query->search($request->get('search'));
                }
                if ($request->has('role_id') && $request->get('role_id') !== 'all') {
                    $query->where('role_id', $request->get('role_id'));
                }
                if ($request->has('status') && $request->get('status') !== 'all') {
                    $query->where('status', $request->get('status'));
                }
            }])
            // Only include societies that match the society_id filter if present
            ->when($request->has('society_id') && $request->get('society_id') !== 'all', function ($q) use ($request) {
                $q->where('id', $request->get('society_id'));
            })
            // Only include societies that HAVE matching users (if filters are applied)
            ->whereHas('users', function ($query) use ($request) {
                if ($request->has('search')) {
                    $query->search($request->get('search'));
                }
                if ($request->has('role_id') && $request->get('role_id') !== 'all') {
                    $query->where('role_id', $request->get('role_id'));
                }
                if ($request->has('status') && $request->get('status') !== 'all') {
                    $query->where('status', $request->get('status'));
                }
            });

        $societies = $societiesQuery->orderBy('name')->paginate(10)->withQueryString();

        // 2. Fetch System Users (No Society) - Only if no society filter is applied
        $systemUsers = [];
        if (!$request->has('society_id') || $request->get('society_id') === 'all') {
            $systemUsersQuery = User::whereNull('society_id')->with('role');
            
            if ($request->has('search')) {
                $systemUsersQuery->search($request->get('search'));
            }
            if ($request->has('role_id') && $request->get('role_id') !== 'all') {
                $systemUsersQuery->where('role_id', $request->get('role_id'));
            }
            if ($request->has('status') && $request->get('status') !== 'all') {
                $systemUsersQuery->where('status', $request->get('status'));
            }
            
            $systemUsers = $systemUsersQuery->get();
        }

        // Lists for filters
        $allSocieties = Society::where('status', 'active')->orderBy('name')->get();
        $roles = Role::orderBy('name')->get();

        return Inertia::render('SuperAdmin/Users/Index', [
            'groupedSocieties' => $societies, // Paginated societies with users
            'systemUsers' => $systemUsers,    // Users without society
            'societies' => $allSocieties,     // For filter dropdown
            'roles' => $roles,
            'filters' => $request->only(['search', 'society_id', 'role_id', 'status']),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): Response
    {
        $societies = Society::where('status', 'active')->orderBy('name')->get();
        $roles = Role::orderBy('name')->get();

        return Inertia::render('SuperAdmin/Users/Create', [
            'societies' => $societies,
            'roles' => $roles,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreUserRequest $request): RedirectResponse
    {
        $data = $request->validated();
        $data['password'] = Hash::make($data['password']);

        User::create($data);

        return redirect()->route('super-admin.users.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user): Response
    {
        $user->load(['society', 'role']);

        return Inertia::render('SuperAdmin/Users/Show', [
            'user' => $user,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user): Response
    {
        $societies = Society::where('status', 'active')->orderBy('name')->get();
        $roles = Role::orderBy('name')->get();

        return Inertia::render('SuperAdmin/Users/Edit', [
            'user' => $user->load(['society', 'role']),
            'societies' => $societies,
            'roles' => $roles,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateUserRequest $request, User $user): RedirectResponse
    {
        $data = $request->validated();

        if (empty($data['password'])) {
            unset($data['password']);
        } else {
            $data['password'] = Hash::make($data['password']);
        }

        $user->update($data);

        return redirect()->route('super-admin.users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user): RedirectResponse
    {
        $user->delete();

        return redirect()->route('super-admin.users.index')
            ->with('success', 'User deleted successfully.');
    }
}
