<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\SystemSetting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class SystemSettingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): Response
    {
        $settings = SystemSetting::all()->groupBy('group');

        return Inertia::render('SuperAdmin/Settings/Index', [
            'settings' => $settings,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request): RedirectResponse
    {
        $data = $request->all();

        // Mapping of setting keys to their groups
        $settingGroups = [
            // Frontend/CMS settings
            'enable_features_section' => 'frontend',
            'enable_testimonials_section' => 'frontend',
            'enable_faq_section' => 'frontend',
            'enable_cta_section' => 'frontend',
            'hero_title' => 'frontend',
            'hero_subtitle' => 'frontend',
            'hero_badge' => 'frontend',
            // Branding settings
            'app_name' => 'branding',
            'app_logo' => 'branding',
            'app_logo_dark' => 'branding',
            'app_favicon' => 'branding',
            'primary_color' => 'branding',
            // General settings
            'enable_landing_page' => 'general',
            'enable_pricing_page' => 'general',
            'enable_registration' => 'general',
            // SEO settings
            'meta_title' => 'seo',
            'meta_description' => 'seo',
            // Payment gateway settings
            'stripe_enabled' => 'payment',
            'razorpay_enabled' => 'payment',
            'paypal_enabled' => 'payment',
            'stripe_key' => 'payment',
            'stripe_secret' => 'payment',
            'razorpay_key' => 'payment',
            'razorpay_secret' => 'payment',
            'razorpay_webhook_secret' => 'payment',
            'paypal_client_id' => 'payment',
            'paypal_secret' => 'payment',
            'paypal_sandbox' => 'payment',
            // Mail settings
            'mail_mailer' => 'mail',
            'mail_host' => 'mail',
            'mail_port' => 'mail',
            'mail_encryption' => 'mail',
            'mail_from_address' => 'mail',
            'mail_from_name' => 'mail',
            // Legal/Content settings
            'privacy_policy' => 'legal',
            'terms_of_service' => 'legal',
            'refund_policy' => 'legal',
            'about_us' => 'legal',
            // Firebase settings
            'firebase_project_id' => 'firebase',
            'firebase_enabled' => 'firebase',
        ];

        foreach ($data as $key => $value) {
            // Skip if value is null (unless we want to explicitly clear it)
            if ($value === null) continue;

            // Get existing setting to preserve group and other fields
            $existingSetting = SystemSetting::where('key', $key)->first();
            
            if ($existingSetting) {
                // Update existing setting, preserve group and other fields
                $existingSetting->update(['value' => $value]);
            } else {
                // Create new setting with appropriate group
                $group = $settingGroups[$key] ?? 'general';
                $type = 'text';
                
                // Determine type based on value
                if (is_bool($value) || $value === '1' || $value === '0' || $value === 'true' || $value === 'false') {
                    $type = 'boolean';
                    $value = is_string($value) ? ($value === 'true' || $value === '1' ? '1' : '0') : ($value ? '1' : '0');
                } elseif (is_numeric($value)) {
                    $type = 'number';
                }
                
                SystemSetting::create([
                    'key' => $key,
                    'value' => $value,
                    'group' => $group,
                    'type' => $type,
                    'description' => ucfirst(str_replace('_', ' ', $key)),
                    'is_public' => in_array($key, ['enable_features_section', 'enable_testimonials_section', 'enable_faq_section', 'enable_cta_section', 'hero_title', 'hero_subtitle', 'hero_badge']),
                ]);
            }
        }

        return back()->with('success', 'Settings updated successfully.');
    }

    /**
     * Upload a file for a setting (e.g., logo).
     */
    public function upload(Request $request): RedirectResponse
    {
        $request->validate([
            'file' => 'required|image|max:2048', // 2MB max
            'key' => 'required|string|exists:system_settings,key',
        ]);

        $file = $request->file('file');
        $key = $request->input('key');

        // Store file in public disk
        $path = $file->store('branding', 'public');

        // Update setting with file path
        SystemSetting::where('key', $key)->update(['value' => '/storage/' . $path]);

        return back()->with('success', 'File uploaded successfully.');
    }

    /**
     * Upload Firebase service account credentials.
     */
    public function uploadFirebaseCredentials(Request $request): RedirectResponse
    {
        $request->validate([
            'firebase_credentials' => 'required|file|mimes:json|max:100', // 100KB max for JSON
        ]);

        $file = $request->file('firebase_credentials');

        // Validate JSON content
        $content = file_get_contents($file->getRealPath());
        $json = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return back()->withErrors(['firebase_credentials' => 'Invalid JSON file.']);
        }

        // Validate required Firebase keys
        $requiredKeys = ['type', 'project_id', 'private_key_id', 'private_key', 'client_email'];
        foreach ($requiredKeys as $key) {
            if (!isset($json[$key])) {
                return back()->withErrors(['firebase_credentials' => "Missing required key: {$key}"]);
            }
        }

        // Ensure the firebase directory exists
        $firebasePath = storage_path('app/firebase');
        if (!is_dir($firebasePath)) {
            mkdir($firebasePath, 0755, true);
        }

        // Store credentials securely (not in public storage)
        $credentialsPath = $firebasePath . '/firebase-credentials.json';
        file_put_contents($credentialsPath, $content);

        // Update system settings
        SystemSetting::updateOrCreate(
            ['key' => 'firebase_project_id'],
            [
                'value' => $json['project_id'],
                'group' => 'firebase',
                'type' => 'text',
                'description' => 'Firebase Project ID',
                'is_public' => false,
            ]
        );

        SystemSetting::updateOrCreate(
            ['key' => 'firebase_enabled'],
            [
                'value' => '1',
                'group' => 'firebase',
                'type' => 'boolean',
                'description' => 'Enable Firebase Push Notifications',
                'is_public' => false,
            ]
        );

        return back()->with('success', 'Firebase credentials uploaded successfully. Push notifications are now enabled.');
    }

    /**
     * Remove Firebase credentials.
     */
    public function removeFirebaseCredentials(): RedirectResponse
    {
        $credentialsPath = storage_path('app/firebase/firebase-credentials.json');

        if (file_exists($credentialsPath)) {
            unlink($credentialsPath);
        }

        // Update system settings
        SystemSetting::where('key', 'firebase_enabled')->update(['value' => '0']);

        return back()->with('success', 'Firebase credentials removed. Push notifications are disabled.');
    }
}
