<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\StoreSocietyRequest;
use App\Http\Requests\SuperAdmin\UpdateSocietyRequest;
use App\Models\Role;
use App\Models\Society;
use App\Models\SocietySetting;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Inertia\Response;

class SocietyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): Response
    {
        try {
            // Calculate statistics
            $totalSocieties = Society::count();
            $activeSocieties = Society::where('status', 'active')->count();
            $pendingSocieties = Society::where('status', 'pending')->count();
            $suspendedSocieties = Society::where('status', 'suspended')->count();
            $totalUsers = User::whereNotNull('society_id')->count();
            
            // Calculate pending fees (societies with pending/trial subscriptions)
            $pendingFees = Society::whereHas('subscription', function ($query) {
                $query->whereIn('status', ['pending', 'trial']);
            })->count();

            $stats = [
                'total_societies' => $totalSocieties,
                'active_societies' => $activeSocieties,
                'pending_societies' => $pendingSocieties,
                'suspended_societies' => $suspendedSocieties,
                'total_users' => $totalUsers,
                'pending_fees' => $pendingFees,
            ];

            $query = Society::query();

            if ($request->has('search')) {
                $query->search($request->get('search'));
            }

            if ($request->has('status') && $request->get('status') !== 'all') {
                $statuses = is_array($request->get('status'))
                    ? $request->get('status')
                    : [$request->get('status')];
                $query->statusFilter($statuses);
            }

            $societies = $query->withCount('users')
                ->with([
                    'admin' => function ($query) {
                        $query->select('id', 'society_id', 'name', 'email', 'phone', 'role_id');
                    },
                    'admin.role' => function ($query) {
                        $query->select('id', 'name', 'display_name');
                    }
                ])
                ->latest()
                ->paginate(15)
                ->withQueryString();

            return Inertia::render('SuperAdmin/Societies/Index', [
                'societies' => $societies,
                'filters' => $request->only(['search', 'status']),
                'stats' => $stats,
            ]);
        } catch (\Exception $e) {
            // Log the error for debugging
            Log::error('Society index error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            
            return Inertia::render('SuperAdmin/Societies/Index', [
                'societies' => Society::query()->latest()->paginate(15)->withQueryString(),
                'filters' => $request->only(['search', 'status']),
                'stats' => [
                    'total_societies' => Society::count(),
                    'active_societies' => 0,
                    'pending_societies' => 0,
                    'suspended_societies' => 0,
                    'total_users' => 0,
                    'pending_fees' => 0,
                ],
                'error' => 'An error occurred while loading societies. Please check the logs.',
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): Response
    {
        $plans = SubscriptionPlan::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        return Inertia::render('SuperAdmin/Societies/Create', [
            'subscriptionPlans' => $plans,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreSocietyRequest $request): RedirectResponse
    {
        try {
            return DB::transaction(function () use ($request) {
                $data = $request->validated();

                // Create society admin user first
                $adminData = $request->input('admin');
                $societyAdminRole = Role::where('name', 'society_admin')->first();

                if (! $societyAdminRole) {
                    return redirect()->back()
                        ->withErrors(['admin' => 'Society Admin role not found. Please seed roles first.'])
                        ->withInput();
                }

                $admin = User::create([
                    'name' => $adminData['name'],
                    'email' => $adminData['email'],
                    'phone' => $adminData['phone'],
                    'password' => Hash::make($adminData['password']),
                    'role_id' => $societyAdminRole->id,
                    'status' => 'active',
                    'directory_visible' => false,
                ]);

                // Set platform_user_id in society data - REMOVED as per cleanup
                // $data['platform_user_id'] = $admin->id;

                $society = Society::create($data);

                // Update admin's society_id after society is created
                $admin->update(['society_id' => $society->id]);

                // Create subscription from selected plan
                $planId = $request->input('subscription.plan_id');
                // Cast to integer if it's a string
                if (is_string($planId)) {
                    $planId = (int) $planId;
                }
                $plan = $planId ? SubscriptionPlan::find($planId) : SubscriptionPlan::where('name', 'basic')->first();

                if (! $plan) {
                    throw new \Exception('Selected plan not found.');
                }

                $billingCycle = $request->input('subscription.billing_cycle', 'monthly');
                $startDate = $request->input('subscription.start_date') ? \Carbon\Carbon::parse($request->input('subscription.start_date')) : now();

                // Auto-calculate end date based on billing cycle if not provided
                $endDate = $request->input('subscription.end_date')
                    ? \Carbon\Carbon::parse($request->input('subscription.end_date'))
                    : match ($billingCycle) {
                        'monthly' => $startDate->copy()->addMonth(),
                        'quarterly' => $startDate->copy()->addMonths(3),
                        'annual' => $startDate->copy()->addYear(),
                        default => $startDate->copy()->addMonth(),
                    };

                // Calculate next billing date based on cycle
                $nextBillingDate = match ($billingCycle) {
                    'monthly' => $startDate->copy()->addMonth(),
                    'quarterly' => $startDate->copy()->addMonths(3),
                    'annual' => $startDate->copy()->addYear(),
                    default => $startDate->copy()->addMonth(),
                };

                $subscriptionData = [
                    'society_id' => $society->id,
                    'plan_id' => $plan->id,
                    'billing_cycle' => $billingCycle,
                    'status' => $request->input('subscription.status', 'trial'),
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'next_billing_date' => $nextBillingDate,
                    'auto_renew' => $request->has('subscription.auto_renew') || ! $request->has('subscription'),
                    // Store plan details for backward compatibility
                    'plan_name' => $plan->name,
                    'plan_display_name' => $plan->display_name,
                    'monthly_price' => $plan->monthly_price,
                    'annual_price' => $plan->annual_price,
                    'max_users' => $plan->max_users,
                    'max_units' => $plan->max_units,
                    'max_buildings' => $plan->max_buildings,
                    'storage_gb' => $plan->storage_gb,
                    'sms_enabled' => $plan->sms_enabled,
                    'whatsapp_enabled' => $plan->whatsapp_enabled,
                    'email_enabled' => $plan->email_enabled,
                    'fcm_enabled' => $plan->fcm_enabled,
                    'advanced_reports' => $plan->advanced_reports,
                    'api_access' => $plan->api_access,
                ];

                // Set trial dates if status is trial
                if ($subscriptionData['status'] === 'trial') {
                    $subscriptionData['trial_start_date'] = $startDate;
                    $subscriptionData['trial_end_date'] = $startDate->copy()->addDays(14);
                }

                $subscription = Subscription::create($subscriptionData);

                // Determine primary payment gateway based on enabled gateways
                $paymentGateway = 'none';
                if ($request->has('settings.razorpay_enabled')) {
                    $paymentGateway = 'razorpay';
                } elseif ($request->has('settings.stripe_enabled')) {
                    $paymentGateway = 'stripe';
                } elseif ($request->has('settings.payu_enabled')) {
                    $paymentGateway = 'payu';
                } elseif ($request->has('settings.cashfree_enabled')) {
                    $paymentGateway = 'cashfree';
                }

                // Create society settings - only enable/disable flags and basic defaults
                // Detailed configuration will be managed by Society Admin later
                $settingsData = [
                    'society_id' => $society->id,
                    'payment_gateway' => $paymentGateway,
                    'razorpay_enabled' => $request->has('settings.razorpay_enabled'),
                    'stripe_enabled' => $request->has('settings.stripe_enabled'),
                    'payu_enabled' => $request->has('settings.payu_enabled'),
                    'cashfree_enabled' => $request->has('settings.cashfree_enabled'),
                    'email_enabled' => $request->has('settings.email_enabled') || ! $request->has('settings'),
                    'sms_enabled' => $request->has('settings.sms_enabled'),
                    'whatsapp_enabled' => $request->has('settings.whatsapp_enabled'),
                    'fcm_enabled' => $request->has('settings.fcm_enabled') || ! $request->has('settings'),
                    'timezone' => $request->input('settings.timezone', 'Asia/Kolkata') ?: 'Asia/Kolkata',
                    'currency' => $request->input('settings.currency', 'INR') ?: 'INR',
                    'date_format' => $request->input('settings.date_format', 'd/m/Y') ?: 'd/m/Y',
                    'time_format' => $request->input('settings.time_format', 'H:i') ?: 'H:i',
                    'language' => $request->input('settings.language', 'en') ?: 'en',
                    'notify_bill_generated' => $request->has('settings.notify_bill_generated') || ! $request->has('settings'),
                    'notify_payment_received' => $request->has('settings.notify_payment_received') || ! $request->has('settings'),
                    'notify_visitor_arrival' => $request->has('settings.notify_visitor_arrival') || ! $request->has('settings'),
                    'notify_maintenance_request' => $request->has('settings.notify_maintenance_request') || ! $request->has('settings'),
                    'notify_announcement' => $request->has('settings.notify_announcement') || ! $request->has('settings'),
                    'enable_visitor_management' => $request->has('settings.enable_visitor_management') || ! $request->has('settings'),
                    'enable_maintenance_tickets' => $request->has('settings.enable_maintenance_tickets') || ! $request->has('settings'),
                    'enable_complaints' => $request->has('settings.enable_complaints') || ! $request->has('settings'),
                    'enable_events' => $request->has('settings.enable_events') || ! $request->has('settings'),
                    'enable_documents' => $request->has('settings.enable_documents') || ! $request->has('settings'),
                    'enable_directory' => $request->has('settings.enable_directory') || ! $request->has('settings'),
                    'enable_notices' => $request->has('settings.enable_notices') || ! $request->has('settings'),
                    'enable_polls' => $request->has('settings.enable_polls'),
                    'enable_marketplace' => $request->has('settings.enable_marketplace'),
                    'require_otp_for_login' => $request->has('settings.require_otp_for_login'),
                    'require_otp_for_payment' => $request->has('settings.require_otp_for_payment'),
                    'session_timeout_minutes' => (int) ($request->input('settings.session_timeout_minutes', 120) ?: 120),
                    'enable_2fa' => $request->has('settings.enable_2fa'),
                ];

                $settings = SocietySetting::create($settingsData);

                return redirect()->route('super-admin.societies.index')
                    ->with('success', 'Society created successfully with subscription and settings.');
            });
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['subscription.plan_id' => $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Society $society)
    {
        try {
            $society->load([
                'admin' => function ($query) {
                    $query->select('id', 'society_id', 'name', 'email', 'phone', 'role_id');
                },
                'admin.role' => function ($query) {
                    $query->select('id', 'name', 'display_name');
                },
                'subscription',
                'settings'
            ]);

            return Inertia::render('SuperAdmin/Societies/Show', [
                'society' => $society,
            ]);
        } catch (\Exception $e) {
            Log::error('Society show error: ' . $e->getMessage(), [
                'society_id' => $society->id,
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->route('super-admin.societies.index')
                ->withErrors(['error' => 'Failed to load society details: ' . $e->getMessage()]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Society $society): Response
    {
        $society->load(['admin', 'subscription', 'settings']);
        
        $plans = SubscriptionPlan::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        return Inertia::render('SuperAdmin/Societies/Edit', [
            'society' => $society,
            'subscriptionPlans' => $plans,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateSocietyRequest $request, Society $society): RedirectResponse
    {
        $society->update($request->validated());

        // Update or create subscription if provided
        if ($request->has('subscription') && $request->input('subscription.plan_id')) {
            $planId = $request->input('subscription.plan_id');
            $plan = SubscriptionPlan::find($planId);

            if ($plan) {
                $billingCycle = $request->input('subscription.billing_cycle', 'monthly');
                $startDate = $request->input('subscription.start_date') 
                    ? \Carbon\Carbon::parse($request->input('subscription.start_date')) 
                    : now();

                $endDate = $request->input('subscription.end_date')
                    ? \Carbon\Carbon::parse($request->input('subscription.end_date'))
                    : match ($billingCycle) {
                        'monthly' => $startDate->copy()->addMonth(),
                        'quarterly' => $startDate->copy()->addMonths(3),
                        'annual' => $startDate->copy()->addYear(),
                        default => $startDate->copy()->addMonth(),
                    };

                $nextBillingDate = match ($billingCycle) {
                    'monthly' => $startDate->copy()->addMonth(),
                    'quarterly' => $startDate->copy()->addMonths(3),
                    'annual' => $startDate->copy()->addYear(),
                    default => $startDate->copy()->addMonth(),
                };

                // Check if existing subscription is expired and being renewed
                $existingSubscription = $society->subscription;
                $isRenewingExpired = $existingSubscription && 
                    ($existingSubscription->isExpired() || 
                     $existingSubscription->status === 'expired' || 
                     $existingSubscription->status === 'trial' ||
                     $existingSubscription->status === 'pending_payment');
                
                // Determine subscription status
                $subscriptionStatus = $request->input('subscription.status');
                
                // If end_date is in the future, ensure status is 'active' (unless explicitly set otherwise)
                if ($endDate > now()->toDateString()) {
                    // Only auto-set to active if:
                    // 1. No status provided, OR
                    // 2. Status is expired/trial/pending_payment (renewing from expired state)
                    if (!$subscriptionStatus || in_array($subscriptionStatus, ['expired', 'trial', 'pending_payment'])) {
                        $subscriptionStatus = 'active';
                    }
                } elseif (!$subscriptionStatus) {
                    // Default to active if no status provided
                    $subscriptionStatus = 'active';
                }

                $subscriptionData = [
                    'plan_id' => $plan->id,
                    'billing_cycle' => $billingCycle,
                    'status' => $subscriptionStatus,
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'next_billing_date' => $nextBillingDate,
                    'auto_renew' => $request->has('subscription.auto_renew'),
                    'plan_name' => $plan->name,
                    'plan_display_name' => $plan->display_name,
                    'monthly_price' => $plan->monthly_price,
                    'annual_price' => $plan->annual_price,
                    'max_users' => $plan->max_users,
                    'max_units' => $plan->max_units,
                    'max_buildings' => $plan->max_buildings,
                    'storage_gb' => $plan->storage_gb,
                    'sms_enabled' => $plan->sms_enabled,
                    'whatsapp_enabled' => $plan->whatsapp_enabled,
                    'email_enabled' => $plan->email_enabled,
                    'fcm_enabled' => $plan->fcm_enabled,
                    'advanced_reports' => $plan->advanced_reports,
                    'api_access' => $plan->api_access,
                ];

                // Clear trial dates if not in trial status
                if ($subscriptionStatus === 'trial') {
                    $subscriptionData['trial_start_date'] = $startDate;
                    $subscriptionData['trial_end_date'] = $startDate->copy()->addDays(14);
                } else {
                    $subscriptionData['trial_start_date'] = null;
                    $subscriptionData['trial_end_date'] = null;
                }

                $society->subscription()->updateOrCreate(
                    ['society_id' => $society->id],
                    $subscriptionData
                );
                
                // Refresh the subscription relationship to ensure latest data is available
                $society->refresh();
                $society->load('subscription');
            }
        }

        // Update or create settings if provided
        if ($request->has('settings')) {
            $paymentGateway = 'none';
            if ($request->has('settings.razorpay_enabled')) {
                $paymentGateway = 'razorpay';
            } elseif ($request->has('settings.stripe_enabled')) {
                $paymentGateway = 'stripe';
            } elseif ($request->has('settings.payu_enabled')) {
                $paymentGateway = 'payu';
            } elseif ($request->has('settings.cashfree_enabled')) {
                $paymentGateway = 'cashfree';
            }

            $society->settings()->updateOrCreate(
                ['society_id' => $society->id],
                $request->input('settings')
            );
        }

        // Handle Admin Details Update
        if ($request->has('admin_name')) {
            $adminData = [
                'name' => $request->input('admin_name'),
                'email' => $request->input('admin_email'),
                'phone' => $request->input('admin_phone'),
            ];

            // Use admin (from users table) as the source of truth
            $adminUser = $society->admin;

            if ($adminUser) {
                // Update existing admin
                $adminUser->update($adminData);
            } else {
                // Create new admin if none exists
                $role = Role::where('name', 'society_admin')->first();
                
                if ($role) {
                    $password = \Illuminate\Support\Str::random(10); // Generate random password
                    
                    User::create([
                        'name' => $request->input('admin_name'),
                        'email' => $request->input('admin_email'),
                        'phone' => $request->input('admin_phone'),
                        'password' => \Illuminate\Support\Facades\Hash::make($password),
                        'role_id' => $role->id,
                        'society_id' => $society->id,
                        'status' => 'active',
                    ]);

                    // TODO: Send email with password to the new admin
                    // Mail::to($request->input('admin_email'))->send(new NewAdminCreated($adminUser, $password));
                }
            }
        }

        return redirect()->route('super-admin.societies.index')
            ->with('success', 'Society updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Society $society): RedirectResponse
    {
        $society->delete();

        return redirect()->route('super-admin.societies.index')
            ->with('success', 'Society deleted successfully.');
    }
}
