<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Society;
use App\Models\User;
use Inertia\Inertia;

class DashboardController extends Controller
{
    /**
     * Display the super admin dashboard.
     */
    public function index(): \Inertia\Response
    {
        // Current period stats
        $totalSocieties = Society::count();
        $activeSocieties = Society::active()->count();
        $pendingFees = Society::pendingFee()->count();
        $totalUsers = User::count();
        $activeUsers = User::where('status', 'active')->count();
        $monthlyRevenue = Society::revenueStats('month')
            ->with('platformFee')
            ->get()
            ->sum(fn ($s) => $s->platformFee?->amount ?? 0);

        // Previous period stats (last month)
        $prevTotalSocieties = Society::where('created_at', '<', now()->subMonth())->count();
        $prevTotalUsers = User::where('created_at', '<', now()->subMonth())->count();
        $prevMonthlyRevenue = Society::revenueStats('month')
            ->with('platformFee')
            ->where('created_at', '<', now()->subMonth())
            ->get()
            ->sum(fn ($s) => $s->platformFee?->amount ?? 0);
        $prevPendingFees = Society::pendingFee()
            ->where('created_at', '<', now()->subMonth())
            ->count();

        // Calculate growth percentages
        $societiesGrowth = $prevTotalSocieties > 0
            ? round((($totalSocieties - $prevTotalSocieties) / $prevTotalSocieties) * 100, 1)
            : ($totalSocieties > 0 ? 100 : 0);

        $usersGrowth = $prevTotalUsers > 0
            ? round((($totalUsers - $prevTotalUsers) / $prevTotalUsers) * 100, 1)
            : ($totalUsers > 0 ? 100 : 0);

        $revenueGrowth = $prevMonthlyRevenue > 0
            ? round((($monthlyRevenue - $prevMonthlyRevenue) / $prevMonthlyRevenue) * 100, 1)
            : ($monthlyRevenue > 0 ? 100 : 0);

        $pendingFeesGrowth = $prevPendingFees > 0
            ? round((($pendingFees - $prevPendingFees) / $prevPendingFees) * 100, 1)
            : 0;

        $stats = [
            'total_societies' => $totalSocieties,
            'active_societies' => $activeSocieties,
            'pending_fees' => $pendingFees,
            'recent_societies' => Society::recent()->count(),
            'total_users' => $totalUsers,
            'active_users' => $activeUsers,
            'monthly_revenue' => $monthlyRevenue,
            // Growth percentages
            'societies_growth' => $societiesGrowth,
            'users_growth' => $usersGrowth,
            'revenue_growth' => $revenueGrowth,
            'pending_fees_growth' => $pendingFeesGrowth,
            'web_admins' => User::webAdmins()->count(),
            'mobile_users' => User::mobileUsers()->count(),
            'recent_logins' => User::recentLogins()->count(),
            'fcm_stats' => [
                'total_mobile_users' => User::mobileUsers()->count(),
                'push_ready_users' => User::mobileUsers()->whereNotNull('fcm_token')->count(),
                'push_coverage' => User::mobileUsers()->count() > 0
                    ? round((User::mobileUsers()->whereNotNull('fcm_token')->count() / User::mobileUsers()->count()) * 100, 2)
                    : 0,
            ],
        ];

        // Get real notifications
        $notifications = [];
        
        // Subscription expiring soon (within 7 days)
        $expiringSocieties = Society::whereHas('subscription', function ($query) {
            $query->where('end_date', '<=', now()->addDays(7))
                  ->where('end_date', '>=', now());
        })->count();
        
        if ($expiringSocieties > 0) {
            $notifications[] = [
                'id' => 'expiring-' . now()->timestamp,
                'title' => 'Subscriptions Expiring Soon',
                'message' => "{$expiringSocieties} " . ($expiringSocieties === 1 ? 'society subscription' : 'societies subscriptions') . " expiring within 7 days",
                'time' => 'Today',
                'unread' => true,
                'type' => 'warning',
            ];
        }

        // Pending platform fees
        if ($pendingFees > 0) {
            $notifications[] = [
                'id' => 'pending-fees-' . now()->timestamp,
                'title' => 'Pending Platform Fees',
                'message' => "{$pendingFees} " . ($pendingFees === 1 ? 'society has' : 'societies have') . " pending platform fees",
                'time' => 'Today',
                'unread' => true,
                'type' => 'warning',
            ];
        }

        // Recent society registrations (last 24 hours)
        $recentSocietiesCount = Society::where('created_at', '>=', now()->subDay())->count();
        if ($recentSocietiesCount > 0) {
            $notifications[] = [
                'id' => 'new-societies-' . now()->timestamp,
                'title' => 'New Society Registrations',
                'message' => "{$recentSocietiesCount} new " . ($recentSocietiesCount === 1 ? 'society' : 'societies') . " registered in the last 24 hours",
                'time' => 'Today',
                'unread' => true,
                'type' => 'success',
            ];
        }

        // Recent users (last 24 hours)
        $recentUsersCount = User::where('created_at', '>=', now()->subDay())->count();
        if ($recentUsersCount > 0) {
            $notifications[] = [
                'id' => 'new-users-' . now()->timestamp,
                'title' => 'New User Signups',
                'message' => "{$recentUsersCount} new " . ($recentUsersCount === 1 ? 'user has' : 'users have') . " joined in the last 24 hours",
                'time' => 'Today',
                'unread' => false,
                'type' => 'success',
            ];
        }

        $recentSocieties = Society::recent()
            ->with(['platformUser', 'platformFee'])
            ->latest()
            ->limit(5)
            ->get();

        $recentUsers = User::with(['society', 'role'])->latest()->limit(5)->get();

        return Inertia::render('SuperAdmin/Dashboard', [
            'stats' => $stats,
            'recentSocieties' => $recentSocieties,
            'recentUsers' => $recentUsers,
            'cron_command' => "* * * * * cd " . base_path() . " && php artisan schedule:run >> /dev/null 2>&1"
        ]);
    }
}
