<?php

namespace App\Http\Controllers\SubAdmin;

use App\Http\Controllers\Controller;
use App\Models\MaintenanceTicket;
use App\Models\Unit;
use App\Models\User;
use App\Models\Visitor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    public function index(Request $request, \App\Models\Society $society): Response
    {
        $user = Auth::user();

        // Ensure user is sub_admin
        if (! $user->hasRole('sub_admin')) {
            abort(403, 'Only Sub-Admin can access this dashboard.');
        }

        // Ensure user belongs to this society (defense in depth - middleware also checks this)
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // KPI 1: Total Residents
        $totalResidents = User::where('society_id', $society->id)
            ->whereHas('role', function ($q) {
                $q->where('name', 'resident');
            })
            ->count();

        // KPI 2: Open Tickets
        $openTickets = MaintenanceTicket::where('society_id', $society->id)
            ->whereIn('status', ['open', 'assigned', 'in_progress'])
            ->count();

        // KPI 3: Today's Visitors
        $todayVisitors = Visitor::where('society_id', $society->id)
            ->whereDate('created_at', today())
            ->count();

        // KPI 4: Assigned Units (units assigned to this sub-admin if applicable)
        // For now, we'll show total units as sub-admin might manage all
        $assignedUnits = Unit::where('society_id', $society->id)->count();

        // KPI 5: Recent Additions (residents added in last 7 days)
        $recentAdditions = User::where('society_id', $society->id)
            ->whereHas('role', function ($q) {
                $q->where('name', 'resident');
            })
            ->where('created_at', '>=', now()->subDays(7))
            ->count();

        // KPI 6: Pending Actions (pending visitors + open tickets)
        $pendingVisitors = Visitor::where('society_id', $society->id)
            ->where('status', 'pending')
            ->count();
        $pendingTickets = MaintenanceTicket::where('society_id', $society->id)
            ->where('status', 'open')
            ->count();
        $pendingActions = $pendingVisitors + $pendingTickets;

        // Recent Activity Feed (Non-financial only)
        $recentTickets = MaintenanceTicket::where('society_id', $society->id)
            ->with(['user:id,name', 'vendor:id,name'])
            ->latest()
            ->limit(5)
            ->get();

        $recentVisitors = Visitor::where('society_id', $society->id)
            ->with(['user:id,name'])
            ->latest()
            ->limit(5)
            ->get();

        $recentResidents = User::where('society_id', $society->id)
            ->whereHas('role', function ($q) {
                $q->where('name', 'resident');
            })
            ->with(['unit:id,unit_no'])
            ->latest()
            ->limit(5)
            ->get();

        return Inertia::render('SubAdmin/Dashboard', [
            'society' => $society->only(['id', 'name', 'code']),
            'kpis' => [
                'total_residents' => $totalResidents,
                'open_tickets' => $openTickets,
                'today_visitors' => $todayVisitors,
                'assigned_units' => $assignedUnits,
                'recent_additions' => $recentAdditions,
                'pending_actions' => $pendingActions,
            ],
            'recentActivity' => [
                'tickets' => $recentTickets->map(function ($ticket) {
                    return [
                        'id' => $ticket->id,
                        'ticket_no' => $ticket->ticket_no,
                        'subject' => $ticket->subject,
                        'status' => $ticket->status,
                        'priority' => $ticket->priority,
                        'user_name' => $ticket->user->name ?? 'N/A',
                        'vendor_name' => $ticket->vendor->name ?? null,
                        'created_at' => $ticket->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'visitors' => $recentVisitors->map(function ($visitor) {
                    return [
                        'id' => $visitor->id,
                        'name' => $visitor->name,
                        'type' => $visitor->type,
                        'status' => $visitor->status,
                        'user_name' => $visitor->user->name ?? 'N/A',
                        'created_at' => $visitor->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'residents' => $recentResidents->map(function ($resident) {
                    return [
                        'id' => $resident->id,
                        'name' => $resident->name,
                        'phone' => $resident->phone,
                        'unit_no' => $resident->unit->unit_no ?? 'N/A',
                        'created_at' => $resident->created_at->format('Y-m-d H:i'),
                    ];
                }),
            ],
        ]);
    }
}
