<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreVisitorRequest;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Visitor;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Inertia\Inertia;
use Inertia\Response;

class VisitorController extends Controller
{
    /**
     * Display a listing of visitors.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Visitor::where('society_id', $society->id)
            ->with(['user:id,name,phone,unit_id', 'user.unit:id,unit_no', 'checkedInBy:id,name']);

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('vehicle_number', 'like', "%{$search}%")
                    ->orWhere('gatepass_code', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by type
        if ($request->has('type') && $request->get('type') !== 'all') {
            $query->where('type', $request->get('type'));
        }

        // Filter by date
        if ($request->has('date_from')) {
            $query->whereDate('entry_time', '>=', $request->get('date_from'));
        }
        if ($request->has('date_to')) {
            $query->whereDate('entry_time', '<=', $request->get('date_to'));
        }

        $visitors = $query->latest('entry_time')->paginate(15)->withQueryString();

        // Get residents for filter
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        // Calculate Stats (Today)
        $today = now()->startOfDay();
        $totalVisitorsToday = Visitor::where('society_id', $society->id)->whereDate('entry_time', $today)->count();
        $checkedInVisitors = Visitor::where('society_id', $society->id)->where('status', 'checked_in')->count();
        $pendingVisitors = Visitor::where('society_id', $society->id)->where('status', 'pending')->count();

        return Inertia::render('SocietyAdmin/Visitors/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'visitors' => $visitors,
            'residents' => $residents,
            'filters' => $request->only(['search', 'status', 'type', 'date_from', 'date_to']),
            'stats' => [
                'total_today' => $totalVisitorsToday,
                'checked_in' => $checkedInVisitors,
                'pending' => $pendingVisitors,
            ],
        ]);
    }

    /**
     * Show the form for creating a new visitor.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Sub-Admin: View only, no create
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        // Get residents
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        return Inertia::render('SocietyAdmin/Visitors/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'residents' => $residents,
        ]);
    }

    /**
     * Store a newly created visitor.
     */
    public function store(StoreVisitorRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Sub-Admin: View only, no store
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        // Convert "none" values to null for optional fields
        if (isset($data['user_id']) && $data['user_id'] === 'none') {
            $data['user_id'] = null;
        }

        // Generate gatepass code if not provided
        if (empty($data['gatepass_code'])) {
            $data['gatepass_code'] = strtoupper(Str::random(8));
        }

        // Set default status based on entry_time
        if (empty($data['status'])) {
            if (empty($data['entry_time']) || now()->isBefore($data['entry_time'])) {
                $data['status'] = 'pending';
            } else {
                $data['status'] = 'approved';
            }
        }

        Visitor::create($data);

        return redirect()->route('society.visitors.index', $society)
            ->with('success', 'Visitor registered successfully.');
    }

    /**
     * Display the specified visitor.
     */
    public function show(Society $society, Visitor $visitor): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $visitor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no show page
        if ($user->hasRole('sub_admin') && ! $user->hasPermission('visitors.view')) {
            abort(403, 'You do not have permission to view visitor details.');
        }

        $visitor->load([
            'user:id,name,phone,email,unit_id',
            'user.unit:id,unit_no,building_id',
            'user.unit.building:id,name',
            'checkedInBy:id,name',
            'society:id,name',
        ]);

        return Inertia::render('SocietyAdmin/Visitors/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'visitor' => $visitor,
        ]);
    }

    /**
     * Check in a visitor.
     */
    public function checkIn(Society $society, Visitor $visitor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $visitor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no actions
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        if ($visitor->status === 'checked_in') {
            return redirect()->back()
                ->with('error', 'Visitor is already checked in.');
        }

        if ($visitor->status === 'checked_out') {
            return redirect()->back()
                ->with('error', 'Visitor has already checked out.');
        }

        $visitor->update([
            'status' => 'checked_in',
            'entry_time' => now(),
            'checked_in_by' => $user->id,
        ]);

        return redirect()->back()
            ->with('success', 'Visitor checked in successfully.');
    }

    /**
     * Check out a visitor.
     */
    public function checkOut(Society $society, Visitor $visitor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $visitor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no actions
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        if ($visitor->status !== 'checked_in') {
            return redirect()->back()
                ->with('error', 'Visitor must be checked in before checking out.');
        }

        $visitor->update([
            'status' => 'checked_out',
            'exit_time' => now(),
        ]);

        return redirect()->back()
            ->with('success', 'Visitor checked out successfully.');
    }

    /**
     * Approve a visitor request.
     */
    public function approve(Society $society, Visitor $visitor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $visitor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no actions
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        if ($visitor->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'Only pending visitors can be approved.');
        }

        $visitor->update([
            'status' => 'approved',
        ]);

        return redirect()->back()
            ->with('success', 'Visitor approved successfully.');
    }

    /**
     * Reject a visitor request.
     */
    public function reject(Society $society, Visitor $visitor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $visitor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no actions
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view visitors.');
        }

        if ($visitor->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'Only pending visitors can be rejected.');
        }

        $visitor->update([
            'status' => 'rejected',
        ]);

        return redirect()->back()
            ->with('success', 'Visitor rejected successfully.');
    }
}
