<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreVendorRequest;
use App\Http\Requests\SocietyAdmin\UpdateVendorRequest;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Inertia\Response;

class VendorController extends Controller
{
    /**
     * Display a listing of vendors.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $vendorRole = Role::where('name', 'vendor')->first();

        if (! $vendorRole) {
            abort(500, 'Vendor role not found. Please seed roles first.');
        }

        $query = User::where('society_id', $society->id)
            ->where('role_id', $vendorRole->id);

        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('company_name', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        $vendors = $query->orderBy('name')->paginate(15)->withQueryString();

        return Inertia::render('SocietyAdmin/Vendors/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'vendors' => $vendors,
            'filters' => $request->only(['search', 'status']),
        ]);
    }

    /**
     * Show the form for creating a new vendor.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        return Inertia::render('SocietyAdmin/Vendors/Create', [
            'society' => $society->only(['id', 'name', 'code']),
        ]);
    }

    /**
     * Store a newly created vendor.
     */
    public function store(StoreVendorRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $vendorRole = Role::where('name', 'vendor')->first();

        if (! $vendorRole) {
            return redirect()->back()
                ->withErrors(['role' => 'Vendor role not found.'])
                ->withInput();
        }

        $data = $request->validated();

        $vendor = User::create([
            'society_id' => $society->id,
            'role_id' => $vendorRole->id,
            'name' => $data['name'],
            'email' => $data['email'] ?? null,
            'phone' => $data['phone'],
            'password' => Hash::make($data['phone']), // default password as phone, can be changed later
            'status' => $data['status'],
            'directory_visible' => false,
        ]);

        // Optional profile fields stored on users table via additional columns if present
        if (isset($data['company_name'])) {
            $vendor->company_name = $data['company_name'];
        }

        if (isset($data['gst_number'])) {
            $vendor->gst_number = $data['gst_number'];
        }

        if (isset($data['address'])) {
            $vendor->address = $data['address'];
        }

        $vendor->save();

        return redirect()->route('society.vendors.index', $society)
            ->with('success', 'Vendor created successfully.');
    }

    /**
     * Display the specified vendor.
     */
    public function show(Society $society, User $vendor): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $vendor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        return Inertia::render('SocietyAdmin/Vendors/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'vendor' => [
                'id' => $vendor->id,
                'name' => $vendor->name,
                'email' => $vendor->email,
                'phone' => $vendor->phone,
                'company_name' => $vendor->company_name ?? null,
                'gst_number' => $vendor->gst_number ?? null,
                'address' => $vendor->address ?? null,
                'status' => $vendor->status,
                'created_at' => $vendor->created_at?->format('Y-m-d H:i') ?? '',
            ],
        ]);
    }

    /**
     * Show the form for editing the specified vendor.
     */
    public function edit(Society $society, User $vendor): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $vendor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        return Inertia::render('SocietyAdmin/Vendors/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'vendor' => [
                'id' => $vendor->id,
                'name' => $vendor->name,
                'email' => $vendor->email,
                'phone' => $vendor->phone,
                'company_name' => $vendor->company_name ?? null,
                'gst_number' => $vendor->gst_number ?? null,
                'address' => $vendor->address ?? null,
                'status' => $vendor->status,
            ],
        ]);
    }

    /**
     * Update the specified vendor.
     */
    public function update(UpdateVendorRequest $request, Society $society, User $vendor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $vendor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $data = $request->validated();

        $vendor->name = $data['name'];
        $vendor->email = $data['email'] ?? null;
        $vendor->phone = $data['phone'];
        $vendor->status = $data['status'];

        $vendor->company_name = $data['company_name'] ?? null;
        $vendor->gst_number = $data['gst_number'] ?? null;
        $vendor->address = $data['address'] ?? null;

        $vendor->save();

        return redirect()->route('society.vendors.index', $society)
            ->with('success', 'Vendor updated successfully.');
    }

    /**
     * Remove the specified vendor from storage.
     */
    public function destroy(Society $society, User $vendor): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $vendor->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $vendor->delete();

        return redirect()->route('society.vendors.index', $society)
            ->with('success', 'Vendor deleted successfully.');
    }
}
